<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use App\Models\TrxCode;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;

class BillingsExport implements FromArray, WithHeadings, WithStyles, WithTitle
{
    protected $payrolls;
    protected $uniqueHeaders;
    protected $client;
    protected $batchCode;
    protected $latestCharge;



    public function __construct($payrolls, $uniqueHeaders, $client, $batchCode, $latestCharge)
    {
        $this->payrolls = $payrolls;
        $this->uniqueHeaders = $uniqueHeaders;
        $this->client = $client;
        $this->batchCode = $batchCode;
        $this->latestCharge = $latestCharge;
    }

    public function array(): array
    {
        $data = [];

        // Populate payroll data
        foreach ($this->payrolls as $payroll) {
            $employeeName = $payroll->employee->first_name . ' ' .
                substr($payroll->employee->middle_name, 0, 1) . '. ' .
                $payroll->employee->last_name;

            $days = $payroll->days_adjustment;
            if (is_null($days)) {
                $days = $payroll->days_worked_withdeduction;
            }

            $rate = $payroll->rate_adjustment;
            if (is_null($rate)) {
                $rate = $payroll->rate;
            }

            // Calculate Amount based on Days and Rate
            $amount = $days * $rate;

            // Initialize the row for each payroll entry
            $row = [
                'Employee Name' => $employeeName,
                'Days' => $days,
                'Rate' => $rate,
                'Amount' => $amount
            ];
            // Add "Others" earnings
            foreach ($this->uniqueHeaders['others'] ?? [] as $header) {
                $others = $payroll->othersForPayroll->firstWhere('name', $header);
                $row[$header] = $others ? $others->amount ?? 0 : 0;
            }

            // Add deductions
            foreach ($this->uniqueHeaders['deductions'] ?? [] as $header) {
                $deduction = $payroll->trxConstantDeductions->firstWhere(
                    fn($earning) => optional(TrxCode::find($earning->trxConstant->group_code_id))->description === $header
                );
                $row[$header] = $deduction ? $deduction->E_Amount ?? 0 : 0;
            }

            // Add benefits
            foreach ($this->uniqueHeaders['benefits'] ?? [] as $header) {
                $benefit = $payroll->benefits->firstWhere('type', $header);
                $row[$header] = $benefit ? $benefit->employer_share ?? 0 : 0;
            }

            // Add Grand Total
            $row['Grand Total'] = array_sum(array_filter($row, fn($value, $key) => is_numeric($value) && !in_array($key, ['Days', 'Rate']), ARRAY_FILTER_USE_BOTH));

            $data[] = $row;
        }

        // Add empty rows for separation (optional)
        $data[] = array_fill_keys(array_keys($data[0] ?? []), '');

        // Add footer totals
        $footerTotals = ['Employee Name' => 'Totals']; // First column label
        foreach (array_keys($data[0] ?? []) as $key) {
            if ($key !== 'Employee Name' && is_numeric($data[0][$key] ?? null)) {
                $footerTotals[$key] = array_sum(array_column($data, $key));
            } else {
                $footerTotals[$key] = ''; // Leave non-numeric columns empty
            }
        }
        $data[] = $footerTotals;



        return $data;
    }
    public function financialSummary(): array
    {
        $dataWithoutFooter = array_slice($this->array(), 0, -1);
        $total_days = array_sum(array_column($dataWithoutFooter, 'Days'));

        $grandTotal = array_sum(array_column($dataWithoutFooter, 'Grand Total'));
        if ($this->latestCharge->service_charge_type == 'percentage') {
            $service_charge = $this->latestCharge->service_charge / 100 * $grandTotal;
            $subtotal = $service_charge + $grandTotal;
            $vat = $subtotal * ($this->latestCharge->vat / 100);
            $subtotal1 = $vat + $subtotal;
            $holding = $subtotal1 * ($this->latestCharge->withholding_tax / 100);
            $Amountbill = $subtotal1 - $holding;

            return [
                ['Service Charge (%)' => $this->latestCharge->service_charge . '%', 'Service Charge' => number_format($service_charge, 2)],
                ['Subtotal' => number_format($subtotal, 2)],
                ['VAT (%)' => $this->latestCharge->vat . '%', 'VAT' => number_format($vat, 2)],
                ['Subtotal1' => number_format($subtotal1, 2)],
                ['Withholding Tax (%)' => $this->latestCharge->withholding_tax . '%', 'Holding Tax' => number_format($holding, 2)],
                ['Amount Bill Payable' => number_format($Amountbill, 2)],
            ];
        } else {
            $service_charge = $total_days * $this->latestCharge->charge_amount;
            $subtotal = $service_charge + $grandTotal;
            $vat = $subtotal * ($this->latestCharge->vat / 100);
            $subtotal1 = $vat + $subtotal;
            $holding = $subtotal1 * ($this->latestCharge->withholding_tax / 100);
            $Amountbill = $subtotal1 - $holding;

            return [
                ['Service Charge (%)' => $this->latestCharge->charge_amount, 'Service Charge' => number_format($service_charge, 2)],
                ['Subtotal' => number_format($subtotal, 2)],
                ['VAT (%)' => $this->latestCharge->vat . '%', 'VAT' => number_format($vat, 2)],
                ['Subtotal1' => number_format($subtotal1, 2)],
                ['Withholding Tax (%)' => $this->latestCharge->withholding_tax . '%', 'Holding Tax' => number_format($holding, 2)],
                ['Amount Bill Payable' => number_format($Amountbill, 2)],
            ];
        }
    }

    public function headings(): array
    {
        // Agency information
        $empty2 = [' ', ' '];
        $agencyName = [' ', 'Agency Name: ' . $this->client->client->agency->name];
        $agencyAddress = [' ', 'Address: ' . $this->client->client->agency->address];
        $agencyPhone = [' ', 'Tel: ' . $this->client->client->agency->contact_phone];
        $empty1 = [' ', ' '];

        // Client information
        $client_name = ['Client Name: ' . $this->client->client->business_name, ' '];
        $address = ['Address: ' . $this->client->client->address, ' '];
        $batch = ['Period Covered: ' . $this->batchCode->pay_period_start . ' - ' . $this->batchCode->pay_period_end, ' '];

        // Main headers
        $secondRow = array_merge(
            ['Employee Name', 'Days', 'Rate', 'Amount'],
            array_keys($this->uniqueHeaders['others'] ?? []),
            array_keys($this->uniqueHeaders['deductions'] ?? []),
            array_keys($this->uniqueHeaders['benefits'] ?? []),
            ['Grand Total']
        );

        return [
            $empty2,
            $agencyName,
            $agencyAddress,
            $agencyPhone,
            $empty1,
            $client_name,
            $address,
            $batch,
            $secondRow,
        ];
    }

    public function drawLogo($sheet)
    {
        $drawing = new Drawing();
        $drawing->setName('Logo');
        $drawing->setDescription('Client Logo');
        $drawing->setPath(public_path($this->client->client->agency->contact_person)); // Assuming logo is stored here
        $drawing->setHeight(60);
        $drawing->setCoordinates('B1');
        $drawing->setWorksheet($sheet);
    }

    public function styles($sheet)
    {
        $columnCount = count($this->headings()[8]); // Count the number of columns
        $lastDataRow = $sheet->getHighestRow(); // Get the last row of data
        $financialSummaryStartRow = $lastDataRow + 2; // Start financial summary two rows below the last data row
        $financialSummary = $this->financialSummary(); // Fetch the financial summary
        $financialSummaryRowCount = count($financialSummary); // Number of financial summary rows
        $financialSummaryEndRow = $financialSummaryStartRow + $financialSummaryRowCount - 1; // Calculate end row for financial summary
        $financialSummaryRange = 'A' . $financialSummaryStartRow . ':' . chr(64 + $columnCount) . $financialSummaryEndRow;

        // Add the logo
        $this->drawLogo($sheet);

        // Write financial summary to the sheet
        foreach ($financialSummary as $index => $row) {
            $sheet->fromArray(array_values($row), null, 'A' . ($financialSummaryStartRow + $index));
        }

        // Style for the logo row
        $logoRange = 'B1:' . chr(64 + $columnCount) . '1';
        $sheet->getStyle($logoRange)->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER);

        // Style for agency info rows
        $agencyInfoRange = 'A2:' . chr(64 + $columnCount) . '4';
        $sheet->getStyle($agencyInfoRange)->getFont()->setName('Poppins')->setSize(11)->setBold(true);
        $sheet->getStyle($agencyInfoRange)->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT);

        // Style for client info rows
        $clientInfoRange = 'A6:' . chr(64 + $columnCount) . '8';
        $sheet->getStyle($clientInfoRange)->getFont()->setName('Poppins')->setSize(10)->setItalic(true);
        $sheet->getStyle($clientInfoRange)->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT);

        // Style for the header row (Row 9)
        $headerRange = 'A9:' . chr(64 + $columnCount) . '9';
        $sheet->getStyle($headerRange)->getFont()->setName('Poppins')->setBold(true)->setSize(12);
        $sheet->getStyle($headerRange)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
        $sheet->getStyle($headerRange)->getBorders()->getBottom()->setBorderStyle(Border::BORDER_THIN);
        $sheet->getStyle($headerRange)->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('F0F0F0');

        // Style for the data rows (up to the last data row)
        $dataRange = 'A9:' . chr(64 + $columnCount) . $lastDataRow;
        $sheet->getStyle($dataRange)->getFont()->setName('Poppins');
        $sheet->getStyle($dataRange)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);

        // Style for the financial summary rows
        $sheet->getStyle($financialSummaryRange)->getFont()->setName('Poppins')->setBold(true);
        $sheet->getStyle($financialSummaryRange)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT);
        $sheet->getStyle($financialSummaryRange)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
        $sheet->getStyle($financialSummaryRange)->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('FFFFCC'); // Light yellow background

        // Styling for the 'Grand Total' column
        $sheet->getStyle(chr(64 + $columnCount) . '9:' . chr(64 + $columnCount) . $lastDataRow)
            ->getFont()->setName('Poppins')->setBold(true);

        // Auto-size columns for better readability
        foreach (range('A', chr(64 + $columnCount)) as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }

        // Adjust row heights
        $sheet->getRowDimension(1)->setRowHeight(60); // Logo row height
        $sheet->getRowDimension(2)->setRowHeight(20); // Agency name
        $sheet->getRowDimension(3)->setRowHeight(20); // Address
        $sheet->getRowDimension(4)->setRowHeight(20); // Phone

        // Adjust row height for financial summary rows
        foreach (range($financialSummaryStartRow, $financialSummaryEndRow) as $rowNumber) {
            $sheet->getRowDimension($rowNumber)->setRowHeight(20); // Adjust row height as needed
        }
    }





    // Title for the sheet
    public function title(): string
    {
        return 'Payroll Billings';
    }
}
