<?php

namespace App\Http\Controllers;

use App\Models\Assign;
use App\Models\Payroll;
use App\Models\Benefits;
use App\Models\CustomBraket;
use Illuminate\Http\Request;
use App\Models\AssignCompany;
use App\Models\CustomBenefit;
use App\Models\CustomChannel;
use Illuminate\Support\Facades\Auth;
use App\Models\EmployeeClientChannel;
use App\Models\AssignCompanyDepartment;
use Yajra\DataTables\Facades\DataTables;

class BenefitsController extends Controller
{
    public function fetchemployeebenefits(Request $request, $id)
    {
        if ($request->ajax()) {
            // Get all payroll records for the employee with the associated client and benefits
            $payrolls = Payroll::with('client.client.agency', 'batchCode', 'benefits')
                ->where('employee_id', $id)
                ->where('status', 0)
                ->get();

            // If no payroll records are found, return empty data response
            if ($payrolls->isEmpty()) {
                return response()->json(['data' => []]);
            }

            // Combine all benefits from all payrolls
            $allBenefits = $payrolls->flatMap(function ($payroll) {
                return $payroll->benefits->map(function ($benefit) use ($payroll) {
                    $benefit->agency_name = $payroll->client && $payroll->client->client && $payroll->client->client->agency
                        ? $payroll->client->client->agency->name
                        : 'No Agency';

                    $benefit->business_name = $payroll->client && $payroll->client->client
                        ? $payroll->client->client->business_name
                        : 'No Company';

                    $benefit->department_name = $payroll->client
                        ? $payroll->client->department_name
                        : 'No Client';

                    // Add batch code information if available
                    $benefit->batch_code = $payroll->batchCode
                        ? \Carbon\Carbon::parse($payroll->batchCode->scope_start)->format('F j, Y') . ' - ' . \Carbon\Carbon::parse($payroll->batchCode->scope_end)->format('F j, Y')
                        : 'No Batch Code';


                    return $benefit;
                });
            });

            // Use DataTables to handle the combined benefits data
            return DataTables::of($allBenefits)
                ->addColumn('client', function ($row) {
                    // Access the added 'client_name' property for each benefit
                    return $row->department_name;
                })
                ->addColumn('company', function ($row) {
                    // Access the added 'company_name' property for each benefit
                    return $row->business_name;
                })
                ->addColumn('agency', function ($row) {
                    // Access the added 'agency_name' property for each benefit
                    return $row->agency_name;
                })
                ->addColumn('batch_code', function ($row) {
                    // Format created_at to a readable format
                    return $row->batch_code;
                })
                ->addColumn('action', function ($row) {
                    // Action buttons for editing and deleting benefits
                    return '<div class="flex items-center justify-center gap-2">
                        <button class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md edit-btn" data-id="' . $row->id . '">
                            <i class="bx bx-edit-alt"></i>
                        </button>
                        <button type="button" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md delete-employee" onclick="openModal(\'deleteModal' . $row->id . '\')">
                            <i class="text-xl bx bx-trash-alt"></i>
                        </button>
                    </div>';
                })
                ->make(true);
        }
    }

    public function storeBenefits(Request $request)
    {
        $request->validate([
            'employee_channel_id' => 'required|integer',
            'type' => 'required|string|in:SSS,Pag-IBIG,PhilHealth',
        ]);

        $exists = CustomBenefit::where('employee_channel_id', $request->employee_channel_id)
            ->where('type', $request->type)
            ->exists();

        if ($exists) {
            return response()->json([
                'message' => 'This benefit has already been assigned to the employee.'
            ], 422); // Unprocessable Entity
        }

        CustomBenefit::create($request->only('employee_channel_id', 'type'));

        return response()->json([
            'message' => 'Benefit assigned successfully.'
        ]);
    }
    public function create()
    {
        $emplpoyee = EmployeeClientChannel::with('employee')
            ->where('status', 1)
            ->get();


        $brackets = CustomBraket::all();

        return view('Admin.CustomBenefits.index', compact('emplpoyee', 'brackets'));
    }


    public function customeemployeelist(Request $request)
    {


        $user = Auth::user();

        $agencyIds = $user->viewable_agency_ids;

        $status = $request->get('status');

        $query = EmployeeClientChannel::with('employee.user', 'department')->where('status', 1);


        if (!empty($agencyIds)) {

            $query->whereHas('department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }

        if ($request->has('office_id') && $request->office_id) {

            $office_id = $request->office_id;
            $query->whereHas('employee', function ($q) use ($office_id) {
                $q->where('office_id', $office_id);
            });
        }

        if ($request->has('store_id') && $request->store_id) {

            $store_id = $request->store_id;
            $query->whereHas('employee', function ($q) use ($store_id) {
                $q->where('store_id', $store_id);
            });
        }


        if ($request->has('pwd_filter')) {
            $pwdFilter = $request->pwd_filter;

            if ($pwdFilter == 'Not PWD') {
                // Filter employees where 'pwd' is NULL
                $query->whereHas('employee', function ($q) {
                    $q->whereNull('pwd');
                });
            } else {
                // Filter employees where 'pwd' equals the selected value
                $query->whereHas('employee', function ($q) use ($pwdFilter) {
                    $q->where('pwd', $pwdFilter);
                });
            }
        }

        if ($request->ajax()) {
            $data = $query->get();

            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">
                    <a href="' . url('admin/pre-profile/' . $row->employee_id) . '" class="flex items-center justify-center w-8 h-8 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>
                    </div>';
                    return $actionBtn;
                })
                ->addColumn('employee_name', function ($row) {
                    return $row->employee->first_name . ' ' . ($row->employee->middle_name ?? '') . ' ' . $row->employee->last_name;
                })
                ->addColumn('client', function ($row) {

                    return $row->department ? $row->department->department_name : 'No Active Client';
                })
                ->addColumn('company', function ($row) {

                    return $row->department->client ? $row->department->client->business_name : 'No Active Client';
                })
                ->addColumn('agency', function ($row) {

                    return $row->department->client->agency ? $row->department->client->agency->name : 'No Active Client';
                })
                

                ->addColumn('profile', function ($row) {
                    // Assuming the profile image URL is stored in $row->profile_image
                    $imageUrl = $row->employee->profile ? asset($row->employee->pprofile) : asset('/default/image.png'); // Replace with your default image path
                    return '<div class="flex items-center justify-center"><img src="' . $imageUrl . '" alt="Profile Image" style="width: 50px; height: 50px; border-radius: 50%;"></div>';
                })

                ->addColumn('status', function ($row) {
                    $status = '<div class="flex items-center justify-center">
                    <span class="w-3 h-3 bg-green-500 rounded-full"></span>
                    <span class="ml-2">Active</span>
                </div>';

                    return $status;
                })



                ->addColumn('bank_status', function ($row) {
                    if ($row->bank_number == null) {
                        $bank_status = 'For Application';
                    } else {
                        $bank_status = 'Submitted';
                    }

                    return $bank_status;
                })


                ->rawColumns(['action', 'action1', 'profile', 'status'])
                ->make(true);
        }
    }
    public function viewcustombenefits(Request $request, $id = null)
    {
        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;



        $query = CustomBenefit::with('customChannels', 'employeeChannel.department', 'employeeChannel.employee');

        if (!empty($id)) {
            $employeeChannelIds = EmployeeClientChannel::where('employee_id', $id)->pluck('id');
            if ($employeeChannelIds->isNotEmpty()) {
                $query->whereIn('employee_channel_id', $employeeChannelIds);
            } else {
                // Early return if no matching employee_channel found
                return DataTables::of(collect())->make(true);
            }
        }

        if (!empty($agencyIds)) {

            $query->whereHas('employeeChannel.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereHas('employeeChannel', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });
            }
        }

        if ($request->ajax()) {
            $data = $query->get();

            return DataTables::of($data)

                ->addColumn('employee_name', function ($row) {
                    return $row->employeeChannel->employee->first_name . ' ' . ($row->employeeChannel->employee->middle_name ?? '') . ' ' . $row->employeeChannel->employee->last_name;
                })
                ->addColumn('client', function ($row) {

                    return $row->employeeChannel->department ? $row->employeeChannel->department->department_name : 'No Active Client';
                })
                ->addColumn('company', function ($row) {

                    return $row->employeeChannel->department->client ? $row->employeeChannel->department->client->business_name : 'No Active Client';
                })
                ->addColumn('agency', function ($row) {

                    return $row->employeeChannel->department->client->agency ? $row->employeeChannel->department->client->agency->name : 'No Active Client';
                })
                ->addColumn('view', function ($row) {
                    return '<i  class="text-xl text-blue-600 cursor-pointer bx bx-show" data-id="' . $row->id . '" title="View Brackets"></i>';
                })

                ->rawColumns(['profile', 'view'])
                ->make(true);
        }
    }

    public function showBrackets($benefitId)
    {
        $benefit = CustomBenefit::find($benefitId);

        if (!$benefit) {
            return response()->json([], 404);
        }

        // Eager load brackets to avoid N+1 issue
        $channels = $benefit->customChannels()->with('bracket')->get();

        // Return array of brackets with their channel IDs
        $result = $channels->map(function ($channel) {
            return [
                'channel_id' => $channel->id,
                'bracket' => $channel->bracket,
            ];
        });

        return response()->json($result);
    }



    public function storeBenefitsBracket(Request $request)
    {
        $request->validate([
            'custom_channel_id' => 'required|exists:custombenefits,id',
            'bracket_mode' => 'required|in:existing,new',
        ]);

        if ($request->bracket_mode === 'existing') {
            $request->validate([
                'existing_bracket_id' => 'required|exists:custombraket,id',
            ]);

            // Just link to existing bracket

            CustomChannel::create([
                'custombenefits_id' => $request->custom_channel_id,
                'custombracket_id' => $request->existing_bracket_id,
            ]);
        } else {
            $request->validate([
                'start_range' => 'required|numeric',
                'end_range' => 'required|numeric|gt:start_range',
                'employee_rate' => 'required|numeric|min:0',
                'employeer_rate' => 'required|numeric|min:0',
            ]);

            // Create new bracket
            $bracket = CustomBraket::create([
                'start_range' => $request->start_range,
                'end_range' => $request->end_range,
                'employee_rate' => $request->employee_rate,
                'employeer_rate' => $request->employeer_rate,
            ]);

            // Assign it to the custom channel
            CustomChannel::create([
                'custombenefits_id' => $request->custom_channel_id,
                'custombracket_id' => $bracket->id,
            ]);
        }

        return response()->json(['message' => 'Bracket assigned successfully']);
    }

    public function deleteBracket($bracketId)
    {
        // Find the bracket by its ID and delete it
        $bracket = CustomChannel::find($bracketId);
        $bracket->delete();
        return response()->json(['message' => 'Bracket deleted successfully.']);
    }

    public function getBrackets()
    {
        $brackets = CustomBraket::get();

        return response()->json($brackets);
    }
}
// Compare this snippet from app/Models/EmployeeClientChannel.php: