<?php

namespace App\Http\Controllers;

use App\Models\BillingAdjustment;
use Illuminate\Http\Request;
use App\Exports\BillingsExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Payroll;
use App\Models\TrxCode;
use App\Models\Department;
use App\Models\Batchcode;
use App\Models\Charges;

class BillingAdjustmentController extends Controller
{
    public function index()
    {
        $billingAdjustments = BillingAdjustment::with('payroll')->get();
        return view('billing_adjustments.index', compact('billingAdjustments'));
    }

    public function create()
    {
        return view('billing_adjustments.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'payroll_id' => 'required|exists:payrolls,id',
            'days_worked' => 'nullable|string',
            'basic_salary' => 'nullable|string',
            'bonus' => 'nullable|string',
            'overtime' => 'nullable|string',
            'deductions' => 'nullable|string',
            'gross_pay' => 'nullable|string',
            'net_pay' => 'nullable|string',
            'total_late_hours' => 'nullable|string',
            'total_undertime_hours' => 'nullable|string',
        ]);

        BillingAdjustment::create($request->all());

        return redirect()->route('billing_adjustments.index')->with('success', 'Billing adjustment created successfully.');
    }

    public function show(BillingAdjustment $billingAdjustment)
    {
        return view('billing_adjustments.show', compact('billingAdjustment'));
    }

    public function edit(BillingAdjustment $billingAdjustment)
    {
        return view('billing_adjustments.edit', compact('billingAdjustment'));
    }

    public function update(Request $request, BillingAdjustment $billingAdjustment)
    {
        $request->validate([
            'days_worked' => 'nullable|string',
            'basic_salary' => 'nullable|string',
            'bonus' => 'nullable|string',
            'overtime' => 'nullable|string',
            'deductions' => 'nullable|string',
            'gross_pay' => 'nullable|string',
            'net_pay' => 'nullable|string',
            'total_late_hours' => 'nullable|string',
            'total_undertime_hours' => 'nullable|string',
        ]);

        $billingAdjustment->update($request->all());

        return redirect()->route('billing_adjustments.index')->with('success', 'Billing adjustment updated successfully.');
    }

    public function destroy(BillingAdjustment $billingAdjustment)
    {
        $billingAdjustment->delete();
        return redirect()->route('billing_adjustments.index')->with('success', 'Billing adjustment deleted successfully.');
    }

    public function exportDetailedPayroll(Request $request, $id, $client_id)
    {
        $latestCharge = Charges::where('department_id', $client_id)->first();
        $batchCode = Batchcode::find($id);

        $payrolls = Payroll::with([
            'trxLoanDeductions.trxLoan.trx_code',
            'trxConstantDeductions.trxConstant.trx_code',
            'othersForPayroll',
            'benefits.client',
        ])
            ->where('code_id', $id)
            ->get();

        $uniqueHeaders = [
            'others' => [],
            'deductions' => [],
            'benefits' => [],
        ];

        foreach ($payrolls as $payroll) {
            foreach ($payroll->othersForPayroll as $earning) {
                $uniqueHeaders['others'][$earning->name] = $earning->name;
            }
            foreach ($payroll->trxConstantDeductions->where('status', 1) as $earning) {
                $description = optional(TrxCode::find($earning->trxConstant->group_code_id))->description;
                $uniqueHeaders['deductions'][$description] = $description;
            }
            foreach ($payroll->benefits as $earning) {
                $uniqueHeaders['benefits'][$earning->type] = $earning->type;
            }
        }

        $client = $client_id ? Department::find($client_id) : null;

        // ✅ Return the BinaryFileResponse directly without modifying headers
        return Excel::download(
            new BillingsExport($payrolls, $uniqueHeaders, $client, $batchCode, $latestCharge),
            'detailed_payroll.xlsx'
        );
    }





}
