<?php

namespace App\Http\Controllers;

use App\Models\Agency;
use App\Models\Client;
use App\Models\Stores;
use Illuminate\Support\Facades\Crypt;
use App\Models\Charges;
use App\Models\User;
use App\Models\AssignCompanyDepartment;
use App\Models\Department;
use App\Models\SettingForPayroll;
use App\Models\SettingForBilling;
use App\Models\Company;
use App\Models\AssignCompany;
use App\Models\Assign;
use App\Models\EmployeeClientChannel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;

use Yajra\DataTables\Facades\DataTables;

class ClientController extends Controller
{
    public function index(Request $request)
    {
        $privileges = $request->get('privileges');

        $user = Auth::user();

        $agencyIds = $user->viewable_agency_ids;

        $query = Client::with('user', 'agency'); // Eager load relationships

        if (!empty($agencyIds)) {
            // First: Filter clients by agencies the user can view
            $query->whereHas('agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            // Then: Get assign IDs for this user across those agencies
            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('company_id')
                ->unique();

            // Apply company restriction only if there are any
            if ($restrictedCompanyIds->isNotEmpty()) {
                $query->whereIn('id', $restrictedCompanyIds);
            }
        }


        if ($request->ajax()) {

            $data = $query->get();





            return DataTables::of($data)
                ->addColumn('action', function ($row) use ($privileges) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">';

                    if (in_array('Manage Company', $privileges)) {
                        $actionBtn .= '<a href="' . url('admin/client-profile/' . $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>';
                    }

                    if (in_array('Edit Company', $privileges)) {
                        $actionBtn .= '<button onclick="openEditModal(' . $row->id . ')" 
                    class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md">
                    <i class="bx bx-edit-alt"></i>
                  </button>';
                    }

                    if (in_array('Delete Company', $privileges)) {
                        $actionBtn .= '<button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                        <i class="text-xl bx bx-trash-alt"></i>
                    </button>';
                    }

                    $actionBtn .= '</div>';
                    return $actionBtn;
                })

                ->make(true);
        }
        $employees = Client::with('user')->get();

        $client = Client::with('user', 'agency')->get();

        $companies = Agency::all();

        return view('Admin.Client.index', compact('employees', 'companies', 'client'));
    }



    public function show($id)
    {
        $client = Client::findOrFail($id);
        return response()->json($client);
    }







    public function showemployeedetails(Request $request, $id)
    {

        $user = Auth::user();

        $agencyIds = $user->viewable_agency_ids;

        $query = EmployeeClientChannel::whereHas('department.client', function ($query) use ($id) {
            $query->where('client_id', $id);
        })
            ->with('department.client', 'employee.user');

        if (!empty($agencyIds)) {

            $query->whereHas('department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);

            }
        }

        if ($request->ajax()) {
            $data = $query->get();
            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">
                    <a href="' . url('admin/pre-profile/' . Crypt::encrypt($row->employee_id)) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>
                     </div>';
                    return $actionBtn;
                })
                ->addColumn('status', function ($row) {
                    if ($row->status == '1') {
                        $status = "Active";
                    } else {
                        $status = "inactive";
                    }
                    return $status;
                })
                ->addColumn('name', function ($row) {
                    return $row->employee->last_name . ' ' . $row->employee->first_name . ' ' . $row->employee->Middle_name;
                })
                ->make(true);
        }
        $client = Client::with('user', 'agency')->find($id);
        $department = Department::where('client_id', $id)->get();


        return view('Admin.Client.clientprofile', compact('client', 'department'));
    }
    public function create()
    {
        return view('employees.create');
    }
    public function store(Request $request)
    {
        $validated = $request->validate([
            'agency_id' => 'required|exists:agencies,id',
            'business_name' => 'required|string|max:255',
            'address' => 'required|string|max:255',
            'contact_number' => 'required|string|max:11',

        ]);

        // Create the user
        $user = User::create([
            'name' => $validated['business_name'], // Use the business name as the user's name
            'email' => strtolower(str_replace(' ', '', $validated['business_name'])) . '@example.com', // Default email
            'password' => Hash::make('password' . uniqid()), // Unique password combining 'password' + unique ID
            'role_id' => 2, // Assign the client role
        ]);

        // Create the client
        $client = Client::create([
            'user_id' => $user->id,
            'agency_id' => $validated['agency_id'],
            'business_name' => $validated['business_name'],
            'address' => $validated['address'],
            'contact_number' => $validated['contact_number'],
        ]);

        // Create charges for the client


        return redirect()->route('clients.index')->with('success', $client->business_name . ' Client and Charges added successfully.');
    }

    public function update(Request $request, Client $client)
    {
        // Validate request data
        $validated = $request->validate([
            'business_name' => 'required|string|max:255',
            'contact_number' => 'required|string|max:20',
            'address' => 'required|string|max:255',
            'agency_id' => 'required|exists:agencies,id',
        ]);

        // Update client
        $client->update($validated);

        return response()->json([
            'success' => true,
            'message' => $client->business_name . ' updated successfully.',
            'data' => $client
        ]);
    }

    // public function update(Request $request, business_name $employee)
    // {
    //     $request->validate([
    //         'name' => 'required',
    //         'address' => 'required',
    //         'contact_number' => 'required',
    //         'date_of_employment' => 'required|date',
    //         'salary' => 'required|numeric',
    //         'position' => 'required',
    //     ]);

    //     $employee->update($request->all());

    //     return redirect()->route('employees.index')->with('success', 'Employee updated successfully.');
    // }

    // // Remove an employee from the database
    public function destroy(Client $client)
    {
        $client->delete();

        return response()->json(['success' => 'Client deleted successfully.']);
    }
}
