<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Agency; // Assuming you have this model
use App\Models\Company; // Assuming you have this model
use App\Models\Client;  // Assuming you have this model
use Illuminate\Http\Request;
use App\Models\EmployeeClientChannel;
use Illuminate\Support\Facades\Auth;
use App\Models\AssignCompanyDepartment;
use App\Models\Assign;
use App\Models\AssignCompany;

class DashboardController extends Controller
{

    public function index()
    {
        $employeeData = Employee::selectRaw('MONTH(created_at) as month, COUNT(id) as total')
            ->groupBy('month')
            ->pluck('total', 'month');

        $agencyData = Agency::selectRaw('MONTH(created_at) as month, COUNT(id) as total')
            ->groupBy('month')
            ->pluck('total', 'month');


        $companyData = Client::selectRaw('MONTH(created_at) as month, COUNT(id) as total')
            ->groupBy('month')
            ->pluck('total', 'month');


        $clientData = Client::selectRaw('MONTH(created_at) as month, COUNT(id) as total')
            ->groupBy('month')
            ->pluck('total', 'month');


        $upcomingBirthdays = Employee::upcomingBirthdays(30); // birthdays in next 30 days

        $user = auth()->user();
        $showBirthdayNotification = false;

        // Only check if user is employee
        if ($user->role_id == 9 && $user->employee) {
            $showBirthdayNotification = $user->employee->hasUpcomingBirthday(30);
        }

        return view('Admin.dashboard', compact('user', 'employeeData', 'showBirthdayNotification', 'upcomingBirthdays', 'agencyData', 'companyData', 'clientData'));
    }
    public function getUpcomingBirthdays(Request $request)
    {


        $user = Auth::user();


        $agencyIds = $user->viewable_agency_ids;

        $status = $request->get('status');

        $query = Employee::with('user', 'channelclient')->where('status', '!=', 'Inactive')
            ->whereHas('channelclient', function ($query) {
                $query->whereNotNull('department_id');
            });

        if (!empty($agencyIds)) {

            $query->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereHas('channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });
            }
        }






        if ($search = $request->get('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%$search%")
                    ->orWhere('last_name', 'like', "%$search%");
            });
        }

        $today = now();
        $in30Days = now()->addDays(30);

        $query = $query->get()->filter(function ($employee) use ($today, $in30Days) {
            $dob = \Carbon\Carbon::parse($employee->date_of_birth);
            $nextBirthday = \Carbon\Carbon::createFromDate($today->year, $dob->month, $dob->day);
            if ($nextBirthday->lessThan($today)) {
                $nextBirthday->addYear();
            }
            return $nextBirthday->between($today, $in30Days);
        });

        $sorted = collect($query)->sortBy(function ($emp) {
            return \Carbon\Carbon::parse($emp->date_of_birth)->format('md');
        });

        $paginated = $sorted->values()->forPage($request->get('page', 1), 10);

        return response()->json([
            'data' => $paginated->map(function ($emp) {
                return [
                    'id' => $emp->id,
                    'encrypted_id' => \Crypt::encrypt($emp->id),
                    'first_name' => $emp->first_name,
                    'last_name' => $emp->last_name,
                    'date_of_birth' => $emp->date_of_birth,
                ];
            }),
            'current_page' => $request->get('page', 1),
            'last_page' => ceil($sorted->count() / 10),
        ]);
    }
    public function upcomingEmploymentEnd(Request $request)
    {
        $today = now();
        $limit = $request->get('limit', 10);
        $search = $request->get('search');
        $page = $request->get('page', 1);




        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;

        $status = $request->get('status');


        $query = EmployeeClientChannel::with('employee', 'department.client.agency')
            ->where('status', 1)
            ->whereNotNull('employment_end_date')
            ->whereBetween('employment_end_date', [$today, $today->copy()->addDays(30)]);

        if (!empty($agencyIds)) {

            $query->whereHas('department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {

                $query->whereIn('department_id', $restrictedDepartmentIds);

            }
        }








        if ($search) {
            $query->whereHas('employee', function ($q) use ($search) {
                $q->where('first_name', 'like', "%$search%")
                    ->orWhere('last_name', 'like', "%$search%");
            });
        }

        $paginator = $query->orderBy('employment_end_date')
            ->paginate($limit, ['*'], 'page', $page);

        $transformed = $paginator->getCollection()->map(function ($item) {
            return [
                'id' => $item->id,
                'encrypted_id' => \Crypt::encrypt($item->employee->id ?? $item->employee_id),
                'employee_id' => $item->employee_id,
                'first_name' => $item->employee->first_name ?? '',
                'last_name' => $item->employee->last_name ?? '',
                'employment_end_date' => $item->employment_end_date,
            ];
        });

        return response()->json([
            'data' => $transformed,
            'current_page' => $paginator->currentPage(),
            'last_page' => $paginator->lastPage(),
        ]);
    }


}
