<?php

namespace App\Http\Controllers;

use App\Models\DtrRequest;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use App\Models\Employee;
use Carbon\Carbon;
class DtrRequestController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();

        $query = DtrRequest::with('employee');

        // If role_id = 9, filter only their requests
        if ($user->role_id == 9) {
            $employee = Employee::where('user_id', $user->id)->first();
            if ($employee) {
                $query->where('employee_id', $employee->id);
            } else {
                $query->whereRaw('1=0'); // no records
            }
        }

        // Status filter (only apply if value is not empty)
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $employees = Employee::all();

        if ($request->ajax()) {
            return DataTables::of($query)
                ->addIndexColumn()
                ->addColumn('actions', function ($row) {
                    return '<div class="flex justify-center"> <button 
                        class="delete-btn px-3 py-1 bg-red-500 text-white rounded text-xs" 
                        data-id="' . $row->id . '">
                        Delete
                    </button></div>
                ';
                })
                ->addColumn('employee', function ($row) {
                    return $row->employee
                        ? $row->employee->first_name . ' ' .
                        ($row->employee->middle_name ?? '') . ' ' .
                        $row->employee->last_name
                        : 'N/A';
                })
                ->addColumn('status', function ($row) use ($user) {
                    // Status badge
                    $badge = '<span class="inline-block px-3 py-1 rounded-full text-xs font-medium ' .
                        ($row->status == 'Approved'
                            ? 'bg-green-100 text-green-700 border border-green-300'
                            : ($row->status == 'Rejected'
                                ? 'bg-red-100 text-red-700 border border-red-300'
                                : 'bg-yellow-100 text-yellow-700 border border-yellow-300')) .
                        '">' . e($row->status) . '</span>';

                    if ($user->role_id == 9) {
                        // Employees only see the badge
                        return $badge;
                    }

                    // Admin/HR: badge + action buttons
                    return '
        <div class="flex flex-col space-y-1">
            ' . $badge . '
            <div class="flex space-x-2">
                <button class="status-btn bg-green-500 hover:bg-green-600 text-white px-2 py-1 rounded-md text-xs transition"
                    data-id="' . $row->id . '" data-status="Approved">
                    ✔ Approve
                </button>
                <button class="status-btn bg-red-500 hover:bg-red-600 text-white px-2 py-1 rounded-md text-xs transition"
                    data-id="' . $row->id . '" data-status="Rejected">
                    ✖ Reject
                </button>
            </div>
        </div>
    ';
                })

                ->editColumn('date', function ($row) {
                    return $row->date ? Carbon::parse($row->date)->format('M d, Y') : '';
                })
                ->rawColumns(['actions', 'status'])
                ->make(true);
        }

        $dtrRequests = $query->get();
        return view('Admin.Attendance.request.dtr', compact('dtrRequests', 'employees'));
    }


    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:Approved,Rejected',
        ]);

        $dtr = DtrRequest::findOrFail($id);
        $dtr->status = $request->status;
        $dtr->save();

        return response()->json(['success' => true, 'message' => 'Status updated successfully.']);
    }
    public function create()
    {
        return view('dtr_requests.create');
    }
    public function store(Request $request)
    {
        $validated = $request->validate([
            'date' => 'required|date',
            'type' => 'required|string|max:255',
            'time' => 'required|date_format:H:i', // HH:MM
            'employee_id' => 'nullable|exists:employees,id'
        ]);

        // Normalize time into H:i:s
        $validated['time'] = $validated['time'] . ':00';

        $user = auth()->user();

        if ($user->role_id == 9) {
            // 🔹 Get employee_id where user_id = auth()->id()
            $employee = Employee::where('user_id', $user->id)->first();

            if (!$employee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee record not found for this user.'
                ], 404);
            }

            $validated['employee_id'] = $employee->id;
        } else {
            // 🔹 For other roles, employee_id must come from the request
            if (!$request->filled('employee_id')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee is required.'
                ], 422);
            }
            $validated['employee_id'] = $request->employee_id;
        }

        $validated['status'] = 'Pending';

        $dtr = DtrRequest::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'DTR request created successfully.',
            'data' => $dtr
        ]);
    }

    public function show(DtrRequest $dtrRequest)
    {
        return view('dtr_requests.show', compact('dtrRequest'));
    }
    public function edit(DtrRequest $dtrRequest)
    {
        return view('dtr_requests.edit', compact('dtrRequest'));
    }
    public function update(Request $request, DtrRequest $dtrRequest)
    {
        $validated = $request->validate([
            'date' => 'required|date',
            'type' => 'required|string|max:255',
            'time' => 'required|date_format:H:i:s',
        ]);

        $dtrRequest->update($validated);

        return redirect()->route('dtr_requests.index')
            ->with('success', 'DTR request updated successfully.');
    }
    public function destroy($id)
    {
        $dtr = DtrRequest::find($id);
        $dtr->delete();

        return response()->json([
            'success' => true,
            'message' => 'DTR deleted successfully.',
            'data' => $dtr
        ]);
    }
}
