<?php

namespace App\Http\Controllers;

use Yajra\DataTables\DataTables;
use App\Models\EmployeeContribution;
use App\Models\Employee;


use Illuminate\Http\Request;

class EmployeeContributionController extends Controller
{

    public function index(Request $request)
    {

        $data = EmployeeContribution::all();
        if ($request->ajax()) {
            $data = EmployeeContribution::with('employee')->get();
            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">
                    <a href="' . url('admin/agency-profile/' . $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-green-600 rounded-md edit"><i class="text-xl bx bx-show"></i></a>
                    <button onclick="openModal(\'updateModal' . $row->id . '\')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-m delete"><i class="bx bx-edit-alt"></i></button>
                    <button onclick="openModal(\'deleteModal' . $row->id . '\')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-m delete"><i class="text-xl bx bx-trash-alt" ></i></button>
                </div>';
                    return $actionBtn;
                })

                ->make(true);
        }
        return view('employee_contributions.index', compact('contributions'));
    }

    public function create()
    {
        $employees = Employee::all();
        return view('employee_contributions.create', compact('employees'));
    }

    public function store(Request $request)
    {
        // Validate the form data
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'monthly_salary' => 'required|numeric',
        ]);

        $monthlySalary = $request->input('monthly_salary');
        $sssContribution = $this->calculateSSS($monthlySalary);
        $philhealthContribution = $this->calculatePhilHealth($monthlySalary);
        $pagibigContribution = $this->calculatePagIbig($monthlySalary);

        // Create and save the new contribution record with calculated values
        EmployeeContribution::create([
            'employee_id' => $request->employee_id,
            'monthly_salary' => $monthlySalary,
            'sss_contribution_employee' => $sssContribution['employee'],
            'sss_contribution_employer' => $sssContribution['employer'],
            'philhealth_contribution_employee' => $philhealthContribution['employee'],
            'philhealth_contribution_employer' => $philhealthContribution['employer'],
            'pagibig_contribution_employee' => $pagibigContribution['employee'],
            'pagibig_contribution_employer' => $pagibigContribution['employer'],
        ]);

        // Redirect back with success message
        return redirect()->route('employee_contributions.index')->with('success', 'Contribution added successfully.');
    }


    public function edit($id)
    {
        $contribution = EmployeeContribution::findOrFail($id);
        $employees = Employee::all();
        return view('employee_contributions.edit', compact('contribution', 'employees'));
    }


    public function update(Request $request, EmployeeContribution $employeeContribution)
    {
        // Validate the form data
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'monthly_salary' => 'required|numeric',
        ]);

        // Dynamic calculations based on monthly salary
        $monthlySalary = $request->input('monthly_salary');

        $sssContribution = $this->calculateSSS($monthlySalary);
        $philhealthContribution = $this->calculatePhilHealth($monthlySalary);
        $pagibigContribution = $this->calculatePagIbig($monthlySalary);

        // Update the contribution record with the calculated values
        $employeeContribution->update([
            'employee_id' => $request->employee_id,
            'monthly_salary' => $monthlySalary,
            'sss_contribution_employee' => $sssContribution['employee'],
            'sss_contribution_employer' => $sssContribution['employer'],
            'philhealth_contribution_employee' => $philhealthContribution['employee'],
            'philhealth_contribution_employer' => $philhealthContribution['employer'],
            'pagibig_contribution_employee' => $pagibigContribution['employee'],
            'pagibig_contribution_employer' => $pagibigContribution['employer'],
        ]);

        // Redirect back with success message
        return redirect()->route('employee_contributions.index')->with('success', 'Contribution updated successfully.');
    }


    public function destroy($id)
    {
        EmployeeContribution::destroy($id);
        return redirect()->route('employee_contributions.index')->with('success', 'Contribution deleted successfully.');
    }


    private function calculateSSS($salary)
    {
        // Example logic, replace with actual SSS table lookup logic
        $sss_employee = $salary * 0.045; // Employee pays 4.5%
        $sss_employer = $salary * 0.09;  // Employer pays 9%

        return [
            'employee' => $sss_employee,
            'employer' => $sss_employer,
        ];
    }


    private function calculatePhilHealth($salary)
    {
        // Example logic, PhilHealth contribution is typically 4.5%, divided between employee and employer
        $total_contribution = $salary * 0.045;
        $employee_share = $total_contribution / 2;
        $employer_share = $total_contribution / 2;

        return [
            'employee' => $employee_share,
            'employer' => $employer_share,
        ];
    }


    private function calculatePagIbig($salary)
    {
        // Pag-IBIG contribution is typically 2% of salary with a cap of PHP 100
        $contribution_rate = 0.02;
        $pagibig_employee = min($salary * $contribution_rate, 100);
        $pagibig_employer = min($salary * $contribution_rate, 100);

        return [
            'employee' => $pagibig_employee,
            'employer' => $pagibig_employer,
        ];
    }
}
