<?php

namespace App\Http\Controllers;

use App\Models\Payroll;
use App\Models\Employee;
use App\Models\Benefits;
use App\Models\Contribution;
use App\Models\EmployeeClientChannel;
use Yajra\DataTables\DataTables;
use App\Models\Attendance;
use Illuminate\Http\Request;
use App\Imports\PtsrImport;
use App\Imports\PayrollSummaryImport;
use App\Models\Ptsr;
use App\Models\Department;
use App\Models\TrxCode;
use App\Models\PtsrTrx;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\Models\AssignCompany;
use App\Models\AssignCompanyDepartment;
use App\Models\Assign;




class PtsrController extends Controller
{


    public function show(Request $request, $id)
    {
        if ($request->ajax()) {
            $data = PtsrTrx::where('ptsr_id', $id)->with(['ptsr', 'code'])->get();

            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    return '
                        <div class="flex items-center justify-center gap-2">
                            <a href="' . route('ptsr.show', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md">
                                <i class="text-xl bx bx-show"></i>
                            </a>
                            <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                                <i class="text-xl bx bx-trash-alt"></i>
                            </button>
                        </div>';
                })
                ->addColumn('name', function ($row) {

                    $name = TrxCode::find($row->code_id);
                    return $name->description;
                })
                ->editColumn('created_at', function ($row) {
                    return $row->created_at ? $row->created_at->format('F d, Y h:i A') : 'N/A';
                })
                ->rawColumns(['action', 'name'])
                ->make(true);
        }

        return view('Admin.PTSR.ptsr_trx', compact('id'));
    }


    public function store(Request $request)
    {
        // Validate the request
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'department_id' => 'required|exists:department,id',
            'under_time' => 'nullable|numeric',
            'total' => 'nullable|numeric',
            'adj' => 'nullable|numeric',
            'ot' => 'nullable|numeric',
            'night' => 'nullable|numeric',
            'special_premium' => 'nullable|numeric',
            'legal_holiday' => 'nullable|numeric',
            'duty' => 'nullable|numeric',
            'number_of_days' => 'nullable|numeric',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'status' => 'required|string',
            'status2' => 'nullable|string',
            'trx_codes' => 'nullable|array', // Allow null
            'trx_codes.*' => 'nullable|exists:trx_codes,id',
            'amounts' => 'nullable|array', // Allow null
            'amounts.*' => 'nullable|numeric|min:0',
        ]);

        // Check if a record with the same employee_id, department_id, start_date, and end_date already exists
        // $existingPtsr = Ptsr::where('employee_id', $request->employee_id)
        //     ->where('department_id', $request->department_id)
        //     ->where('start_date', $request->start_date)
        //     ->where('end_date', $request->end_date)
        //     ->exists();

        // if ($existingPtsr) {
        //     return response()->json([
        //         'message' => 'A record with the same Employee, Department, Start Date, and End Date already exists.',
        //     ], 422);
        // }

        // Create the PTSR record
        $ptsr = Ptsr::create($request->only([
            'employee_id',
            'department_id',
            'under_time',
            'total',
            'adj',
            'ot',
            'night',
            'special_premium',
            'legal_holiday',
            'duty',
            'number_of_days',
            'start_date',
            'end_date',
            'status',
            'status2'
        ]));

        // Store PtsrTrx records only if trx_codes exist
        if ($request->has('trx_codes') && is_array($request->trx_codes)) {
            foreach ($request->trx_codes as $index => $codeId) {
                if (!empty($codeId)) {
                    PtsrTrx::create([
                        'ptsr_id' => $ptsr->id,
                        'code_id' => $codeId,
                        'amount' => $request->amounts[$index] ?? 0, // Default to 0 if not provided
                        'type' => $ptsr->status,
                    ]);
                }
            }
        }

        return response()->json([
            'message' => '✅ PTSR record saved successfully!',
            'data' => $ptsr->load('transactions'),
        ], 201);
    }






    public function index(Request $request)
    {
        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;

        $query = ptsr::where('status', 'ptsr')->with('employee.channelclient')
            ->latest('created_at');

        if (!empty($agencyIds)) {
            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });




            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }

        $data = $query->get();

        if ($request->ajax()) {
            return DataTables::of($data)
                ->addColumn('s_date', function ($row) {
                    return $row->start_date;
                })
                ->addColumn('e_date', function ($row) {
                    return $row->end_date;
                })
                ->addColumn('action', function ($row) {
                    return '
                    <div class="flex items-center justify-center gap-2">
                        <a href="' . route('ptsr.show', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md">
                            <i class="text-xl bx bx-show"></i>
                        </a>
                        <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                            <i class="text-xl bx bx-trash-alt"></i>
                        </button>
                    </div>';
                })
                ->addColumn('department', function ($row) {
                    $department = Department::find($row->department_id); // Use department_id instead of row id
                    return $department ? $department->department_name : 'N/A'; // Handle null cases
                })



                ->addColumn('status', function ($row) {
                    return '
                        <input type="checkbox" class="status-checkbox" data-id="' . $row->id . '" 
                            ' . ($row->status2 == 1 ? 'checked' : '') . ' 
                            title="Toggle status">
                    ';
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
        }

        $querydepartment = Department::with('client.agency'); // Define $query before using it

        if (!empty($agencyIds)) {
            $querydepartment->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $querydepartment->whereIn('id', $restrictedDepartmentIds);

            }
        }

        $client = $querydepartment->get(); // Execute the query after applying conditions

        $queryemployee = Employee::whereHas('channelclient', function ($query) {
            $query->whereNotNull('department_id');
        }); // Remove the incorrect query() method

        // Apply additional filtering if $agencyIds is not empty
        if (!empty($agencyIds)) {
            $queryemployee->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });




            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $queryemployee->whereHas('channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });

            }
        }

        // Fetch the employees after all conditions are applied
        $employees = $queryemployee->get();


        $trxCodes = TrxCode::get();

        return view('Admin.PTSR.index', compact('client', 'employees', 'trxCodes'));
    }



    public function updateStatusToNull(Request $request, $id)
    {
        $ptsr = Ptsr::findOrFail($id);

        $ptsr->status2 = $request->status == 1 ? 1 : null;
        $ptsr->save();

        return redirect()->back()->with('success', 'Status updated successfully');
    }


    public function destroy($id)
    {
        $ptsr = Ptsr::findOrFail($id);
        $ptsr->delete();

        return response()->json(['success' => true, 'message' => 'Record deleted successfully.']);
    }
    public function destroy_ptsr_trx($id)
    {
        $ptsr = PtsrTrx::findOrFail($id);
        $ptsr->delete();

        return response()->json(['success' => true, 'message' => 'Record deleted successfully.']);
    }


    public function summary(Request $request)
    {

        $user = Auth::user();

        // Get the agency IDs the user is assigned to
        $agencyIds = $user->viewable_agency_ids;

        $query = ptsr::where('status', 'summary')->with('employee.channelclient');

        if (!empty($agencyIds)) {
            $query->whereHas('employee.channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });




            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereHas('employee.channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });

            }
        }


        $data = $query->get();

        if ($request->ajax()) {
            return DataTables::of($data)
                ->addColumn('s_date', function ($row) {
                    return $row->start_date;
                })
                ->addColumn('e_date', function ($row) {
                    return $row->end_date;
                })
                ->addColumn('action', function ($row) {
                    return '
                    <div class="flex items-center justify-center gap-2">
                        <a href="' . route('ptsr.show', $row->id) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md">
                            <i class="text-xl bx bx-show"></i>
                        </a>
                        <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                            <i class="text-xl bx bx-trash-alt"></i>
                        </button>
                    </div>';
                })


                ->addColumn('status', function ($row) {
                    $csrfField = csrf_field();
                    $updateStatusButton = '';

                    // Check if status is not null and is 1
                    if ($row->status2 == 1) {
                        $updateStatusButton = '
                        <form action="' . route('ptsr.update-status', $row->id) . '" method="POST" style="display:inline;">
                            ' . $csrfField . '
                            <input type="hidden" name="status" value="0"> <!-- Default unchecked value -->
                            <input type="checkbox" name="status" value="1" class="status-checkbox" onchange="this.form.submit()" title="Uncheck to set status to null" checked>
                        </form>';
                    } else {
                        $updateStatusButton = '
                        <form action="' . route('ptsr.update-status', $row->id) . '" method="POST" style="display:inline;">
                            ' . $csrfField . '
                            <input type="hidden" name="status" value="0"> <!-- Default unchecked value -->
                            <input type="checkbox" name="status" value="1" class="status-checkbox" onchange="this.form.submit()" title="Check to set status to 1">
                        </form>';
                    }


                    return '
                    <div class="flex items-center justify-center gap-2">
                        ' . $updateStatusButton . '
                       
                    </div>';
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
        }

        $querydepartment = Department::with('client.agency'); // Define $query before using it

        if (!empty($agencyIds)) {
            $querydepartment->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $querydepartment->whereIn('id', $restrictedDepartmentIds);

            }
        }

        $client = $querydepartment->get(); // Execute the query after applying conditions

        $queryemployee = Employee::whereHas('channelclient', function ($query) {
            $query->whereNotNull('department_id');
        }); // Remove the incorrect query() method

        // Apply additional filtering if $agencyIds is not empty
        if (!empty($agencyIds)) {
            $queryemployee->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });




            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            // Get the company IDs restricted via assignCompanies
            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', $restrictedCompanyIds)
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $queryemployee->whereHas('channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });

            }
        }

        // Fetch the employees after all conditions are applied
        $employees = $queryemployee->get();


        $trxCodes = TrxCode::get();

        return view('Admin.PTSR.summary', compact('client', 'employees', 'trxCodes'));
    }





    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv,xls',
            'department_id' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date',
        ]);

        try {
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');
            $department_id = $request->input('department_id');
            // Perform the import
            $import = new PtsrImport($startDate, $endDate, $department_id);
            Excel::import($import, $request->file('file'));

            return response()->json([
                'success' => true,
                'message' => 'PTS Records imported successfully.',
                'failedRows' => $import->failedRows, // Pass failed rows
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error importing file: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function importsummary(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv,xls',
            'department_id' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date',
        ]);

        try {
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');
            $department_id = $request->input('department_id');
            // Perform the import
            $import = new PayrollSummaryImport($startDate, $endDate, $department_id);
            Excel::import($import, $request->file('file'));

            return response()->json([
                'success' => true,
                'message' => 'PTS Records imported successfully.',
                'failedRows' => $import->failedRows, // Pass failed rows
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error importing file: ' . $e->getMessage(),
            ], 500);
        }
    }
}
