<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Assign;
use App\Models\Attendance;
use App\Models\Department;
use App\Models\Employee;
use Faker\Core\Barcode;
use Illuminate\Http\Request;
use App\Models\AssignCompany;
use Illuminate\Support\Facades\Auth;
use App\Models\EmployeeClientChannel;
use App\Models\Batchcode;
use App\Models\AssignCompanyDepartment;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\BenefitsExport;
use ZipArchive;
use App\Models\Payroll;
use App\Models\TrxCode;
use App\Models\Benefits;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\PayrollReportExport;
class ReportsController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $agencyIds = $user->viewable_agency_ids;
        $querydepartment = Department::with('client'); // Define $query before using it
        $batches = Batchcode::get();
        $query = Employee::whereHas('channelclient', function ($query) {
            $query->whereNotNull('department_id');
        });
        if (!empty($agencyIds)) {
            //department
            $querydepartment->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });
            //employee
            $query->whereHas('channelclient.department.client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $querydepartment->whereIn('id', $restrictedDepartmentIds);
                $query->whereHas('channelclient', function ($q) use ($restrictedDepartmentIds) {
                    $q->whereIn('department_id', $restrictedDepartmentIds);
                });
            }
        }
        $client = $querydepartment->get();
        $employees = $query->get();

        return view('Admin.Reports.index', compact('client', 'batches', 'employees'));

    }
    public function exportPdf(Request $request)
    {
        $departmentId = $request->department_id;
        $startDate = $request->startDate;
        $endDate = $request->endDate;

        // Get all employees in the department
        $employees = EmployeeClientChannel::with('employee', 'department')->where('department_id', $departmentId)->get();

        // If no employees found, return error
        if ($employees->isEmpty()) {
            return back()->with('error', 'No employees found in this department.');
        }

        // Temporary folder to store PDFs
        $tempPath = storage_path('app/public/temp_pdfs');
        if (!file_exists($tempPath)) {
            mkdir($tempPath, 0777, true);
        }

        $pdfFiles = [];

        foreach ($employees as $employee) {
            $query = Attendance::where('employee_id', $employee->employee_id)
                ->where('department_id', $departmentId);

            if ($startDate) {
                $query->whereDate('date', '>=', $startDate);
            }
            if ($endDate) {
                $query->whereDate('date', '<=', $endDate);
            }

            $attendances = $query->orderBy('date')->get();

            // Skip if no attendance
            if ($attendances->isEmpty()) {
                continue;
            }

            // Get employee's full name safely
            $fullName = trim(
                ($employee->employee->first_name ?? '') . ' ' .
                ($employee->employee->middle_name ?? '') . ' ' .
                ($employee->employee->last_name ?? '')
            );

            // If full name is empty, use employee ID as fallback
            if (empty($fullName)) {
                $fullName = 'employee_' . $employee->employee_id;
            }

            // Replace spaces and unsafe characters for file name
            $safeName = preg_replace('/[^A-Za-z0-9_-]/', '_', $fullName);

            $pdf = PDF::loadView(
                'Admin.Attendance.complete.dtr_report',
                compact('attendances', 'employee', 'startDate', 'endDate')
            );

            $pdfFileName = 'attendance_' . $safeName . '.pdf';
            $pdfFilePath = $tempPath . '/' . $pdfFileName;
            $pdf->save($pdfFilePath);

            $pdfFiles[] = $pdfFilePath;
        }

        // If more than 1 PDF, zip them
        if (count($pdfFiles) > 1) {
            // Get department name safely
            $departmentName = $employees->first()->department->department_name ?? 'department';
            $safeDeptName = preg_replace('/[^A-Za-z0-9_-]/', '_', $departmentName);

            $zipFileName = 'attendance_reports_' . $safeDeptName . '.zip';
            $zipFilePath = $tempPath . '/' . $zipFileName;

            $zip = new ZipArchive;
            if ($zip->open($zipFilePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === true) {
                foreach ($pdfFiles as $file) {
                    $zip->addFile($file, basename($file));
                }
                $zip->close();
            }

            // Clean up individual PDFs
            foreach ($pdfFiles as $file) {
                unlink($file);
            }

            return response()->download($zipFilePath)->deleteFileAfterSend(true);
        }

        // Only 1 PDF
        return response()->download($pdfFiles[0])->deleteFileAfterSend(true);
    }

    public function downloadPayrollReport(Request $request)
    {
        // Get batchId, clientId, and format from request
        $batchId = $request->input('batchId');
        $clientId = $request->input('clientId', null);
        $format = $request->input('format', 'pdf'); // default is PDF

        if (!$batchId) {
            return back()->with('error', 'Batch ID is required.');
        }

        // Increase execution time and memory limit
        ini_set('max_execution_time', 300);
        ini_set('memory_limit', '1024M');

        $batchCode = Batchcode::find($batchId);

        // Use chunking to avoid loading all payrolls at once
        $payrolls = collect();
        Payroll::with([
            'employee',
            'trxLoanDeductions.trxLoan.trx_code',
            'trxConstantDeductions.trxConstant.trx_code',
            'othersForPayroll',
            'benefits.client',
        ])
            ->where('code_id', $batchId)
            ->chunk(50, function ($chunk) use (&$payrolls) {
                $payrolls = $payrolls->merge($chunk);
            });

        $trxCodeIds = $payrolls->pluck('trxConstantDeductions.*.trxConstant.group_code_id')->flatten()->unique();
        $trxCodes = TrxCode::whereIn('id', $trxCodeIds)->pluck('description', 'id');

        $standardBenefits = ['SSS', 'Pag-IBIG', 'PhilHealth'];

        $payrollData = $payrolls->map(function ($payroll) use ($standardBenefits, $trxCodes) {
            $gross = $payroll->basic_salary + $payroll->othersForPayroll->sum('amount');
            $deductions = $payroll->trxConstantDeductions->sum('E_Amount');
            $loans = $payroll->trxLoanDeductions->sum('Amount');

            $benefits = $payroll->benefits->filter(fn($b) => in_array($b->type, $standardBenefits))
                ->map(fn($b) => ['type' => $b->type, 'amount' => $b->employee_share])
                ->values();

            $nonStandardDeductions = $payroll->benefits->filter(fn($b) => !in_array($b->type, $standardBenefits))
                ->map(fn($b) => ['type' => $b->type, 'amount' => $b->employee_share])
                ->values();

            return [
                'payroll' => $payroll,
                'gross' => $gross,
                'deductions' => $deductions,
                'loans' => $loans,
                'benefits' => $benefits,
                'nonStandardDeductions' => $nonStandardDeductions,
            ];
        });

        $uniqueHeaders = [
            'others' => $payrolls->flatMap(fn($p) => $p->othersForPayroll->pluck('name'))->unique()->values()->all(),
            'deductions' => $payrolls->flatMap(fn($p) => $p->trxConstantDeductions->pluck('trxConstant.group_code_id'))
                ->map(fn($id) => $trxCodes[$id] ?? 'Unknown')
                ->unique()
                ->values()
                ->all(),
            'benefits' => $payrolls->flatMap(fn($p) => $p->benefits->pluck('type'))->unique()->values()->all(),
        ];

        $client = Department::with('client.agency')->find($batchCode->department_id);

        // Decide output format
        if ($format === 'excel') {
            return Excel::download(
                new PayrollReportExport($batchCode, $payrollData, $uniqueHeaders, $client),
                'payroll_report.xlsx'
            );
        }

        // Default to PDF
        $pdf = Pdf::loadView('Admin.Payroll.payrollbatch.reports', compact('batchCode', 'payrollData', 'uniqueHeaders', 'client'));
        return $pdf->download('payroll_report.pdf');
    }
    public function employeeDepartmentReport(Request $request)
    {
        $query = Employee::with(['channelclient.department']);

        // Apply filter if department is selected
        if ($request->filled('department_id')) {
            $query->whereHas('channelclient', function ($q) use ($request) {
                $q->where('department_id', $request->department_id);
            });
        }

        $employees = $query->get();
        $departments = Department::all();

        // Handle export
        if ($request->has('export')) {
            if ($request->export === 'pdf') {
                $pdf = PDF::loadView('Admin.Reports.employee_department_pdf', compact('employees'))->setPaper('a4', 'landscape');
                return $pdf->download('employee_department_report.pdf');
            }
            if ($request->export === 'excel') {
                return Excel::download(
                    new \App\Exports\EmployeeDepartmentExport($request->department_id),
                    'employee_department_report.xlsx'
                );
            }
        }
    }
    public function beneiftsreports(Request $request)
    {
        $query = Benefits::with('payroll.employee', 'payroll.batchCode');

        // Date range filter (scope by pay_period_start/end in batchcode)
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereHas('payroll.batchCode', function ($q) use ($request) {
                $q->whereDate('pay_period_start', '>=', $request->start_date)
                    ->whereDate('pay_period_end', '<=', $request->end_date);
            });
        }

        // Optional employee filter
        if ($request->filled('employee_id')) {
            $query->whereHas('payroll', function ($q) use ($request) {
                $q->where('employee_id', $request->employee_id);
            });
        }

        // Optional type filter
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        $benefits = $query->get();

        // Export handling
        if ($request->export === 'excel') {
            return Excel::download(new BenefitsExport($benefits), 'benefits_report.xlsx');
        }

        if ($request->export === 'pdf') {
            $pdf = PDF::loadView('Admin.Reports.benefits_pdf', [
                'benefits' => $benefits,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'employee' => $request->employee_id ? Employee::find($request->employee_id) : null,
                'type' => $request->type,
            ])->setPaper('a4', 'landscape');

            return $pdf->download('benefits_report.pdf');
        }

    }


}
