<?php

namespace App\Http\Controllers;

use App\Models\Requisition;
use App\Models\Client;
use App\Models\Agency;
use App\Models\Company;
use App\Models\Employee;
use App\Models\Offices;
use App\Models\Stores;
use App\Models\MrfAction;

use App\Models\AssignCompany;
use App\Models\AssignCompanyDepartment;
use App\Models\Assign;

use App\Models\Department;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use App\Models\MrfApplicant;
use App\Models\MrfMoreDetail;
use App\Models\ApplicantLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
class RequisitionController extends Controller
{
    public function downloadmrf(Requisition $requisition)
    {
        // Load relationships
        $requisition->load('client.agency', 'client', 'department', 'mrfapplicant');

        // Related data collections (if needed in the PDF)
        $agencies = Agency::all();
        $companies = Company::all();
        $clients = Client::all();
        $departments = Department::all();
        $stores = Stores::all();
        $offices = Offices::all();

        $employees = Employee::with('channelclient')
            ->where(function ($query) {
                $query->where('status', 3)
                    ->orWhereDoesntHave('channelclient');
            })
            ->get();

        $mrf = MrfAction::with('user', 'requisition')
            ->where('mrf_id', $requisition->id)
            ->get();

        $details = MrfMoreDetail::where('mrf_id', $requisition->id)->get();

        // Optional: include current user if needed in the PDF
        $user = auth()->user();

        // Generate the PDF with the correct data
        $pdf = Pdf::loadView('Admin.Requisition.parts.pdf', compact(
            'requisition',
            'agencies',
            'companies',
            'clients',
            'departments',
            'stores',
            'offices',
            'employees',
            'mrf',
            'details',
            'user'
        ));

        return $pdf->download('MRF.pdf');
    }

    public function index(Request $request)
    {
        $privileges = $request->get('privileges');

        $user = Auth::user();
        $agencyIds = $user->viewable_agency_ids;


        $query = Requisition::with('client.agency', 'department')->orderBy('created_at', 'desc');

        if ($user->role_id === 8) {
            $query->whereHas('department', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            });
        }

        if (!empty($agencyIds)) {

            $query->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('company_id')
                ->unique();

            if ($restrictedCompanyIds->isNotEmpty()) {
                $query->whereIn('client_id', $restrictedCompanyIds);
            }

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn('assigncompany_id', AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('id'))
                ->pluck('department_id')
                ->unique();


            // Apply company restriction only if there are any
            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('department_id', $restrictedDepartmentIds);
            }
        }

        $query->where(function ($q) use ($privileges) {
            if (in_array('MRF Disapproved', $privileges)) {
                $q->orWhere('status', '!=', 'Disapproved');
            }
            if (in_array('MRF Cancel', $privileges)) {
                $q->orWhere('status', '!=', 'Cancel');
            }
            if (in_array('MRF Open', $privileges)) {
                $q->orWhereNull('status')->orWhere('status', 'Open')->orWhere('status', 'For Approval')->orWhere('status', 'Deploy')->orWhere('status', 'Approved')->orWhere('status', 'Disapproved')->orWhere('status', 'Meet & Greet / Intro')->orWhere('status', 'Process Requirement')->orWhere('status', 'Agency Orientation')->orWhere('status', 'Client Orientation');
                ;
            }
            if (in_array('MRF For Approval', $privileges)) {
                $q->orWhere('status', 'For Approval')->orWhere('status', 'Deploy')->orWhere('status', 'Approved')->orWhere('status', 'Disapproved')->orWhere('status', 'Meet & Greet / Intro')->orWhere('status', 'Process Requirement')->orWhere('status', 'Agency Orientation')->orWhere('status', 'Client Orientation');
            }
            if (in_array('MRF Approved', $privileges)) {
                $q->orWhere('status', 'Approved')->orWhere('status', 'Deploy')->orWhere('status', 'Disapproved')->orWhere('status', 'Meet & Greet / Intro')->orWhere('status', 'Process Requirement')->orWhere('status', 'Agency Orientation')->orWhere('status', 'Client Orientation');
            }
            if (in_array('MRF Meet & Greet', $privileges)) {
                $q->orWhere('status', 'Disapproved')->orWhere('status', 'Deploy')->orWhere('status', 'Meet & Greet / Intro')->orWhere('status', 'Process Requirement')->orWhere('status', 'Agency Orientation')->orWhere('status', 'Client Orientation');
            }
            if (in_array('MRF Process Requirement', $privileges)) {
                $q->orWhere('status', 'Disapproved')->orWhere('status', 'Deploy')->orWhere('status', 'Process Requirement')->orWhere('status', 'Agency Orientation')->orWhere('status', 'Client Orientation');
            }
            if (in_array('MRF Agency Orientation', $privileges)) {
                $q->orWhere('status', 'Disapproved')->orWhere('status', 'Deploy')->orWhere('status', 'Agency Orientation')->orWhere('status', 'Client Orientation');
            }
            if (in_array('MRF Client Orientation', $privileges)) {
                $q->orWhere('status', 'Disapproved')->orWhere('status', 'Deploy')->orWhere('status', 'Client Orientation');
            }
        });

        if ($request->ajax()) {
            return DataTables::of($query)
                ->addColumn('department', function ($row) {
                    return $row->department->department_name ?? 'N/A';
                })
                ->addColumn('acceptbutton', function ($row) use ($privileges) {
                    $statusOptions = [
                        'Open' => 'MRF Open',
                        'For Approval' => 'MRF For Approval',
                        'Approved' => 'MRF Approved',
                        'Meet & Greet / Intro' => 'MRF Meet & Greet',
                        'Process Requirement' => 'MRF Process Requirement',
                        'Agency Orientation' => 'MRF Agency Orientation',
                        'Client Orientation' => 'MRF Client Orientation',
                        'Deploy' => 'MRF Deploy',
                        'Cancel' => 'MRF Cancel',
                        'Disapproved' => 'MRF Disapproved',
                        'Back out' => 'MRF Back Out', // Always allowed
                    ];

                    $selectId = 'status-select-' . $row->id;
                    $hasAccess = false;

                    $select = '<select id="' . $selectId . '" data-id="' . $row->id . '" class="status-select border border-gray-300 rounded-md p-1">';
                    $select .= '<option disabled selected>Change status</option>';

                    foreach ($statusOptions as $status => $requiredPrivilege) {
                        if ($requiredPrivilege === null || in_array($requiredPrivilege, $privileges)) {
                            $hasAccess = true;
                            $selected = $row->status === $status ? 'selected' : '';
                            $select .= '<option value="' . $status . '" ' . $selected . '>' . $status . '</option>';
                        }
                    }

                    $select .= '</select>';

                    // Button calls showStatusModalDynamic with the requisition id
                    $button = '<button onclick="showStatusModalDynamic(' . $row->id . ')" class="ml-2 px-3 py-1 text-white bg-blue-600 rounded hover:bg-blue-800">Save</button>';

                    if ($hasAccess) {
                        return '<div class="flex items-center justify-center">' . $select . $button . '</div>';
                    } else {
                        return '<span class="text-gray-600 font-semibold">' . htmlspecialchars($row->status ?? 'N/A') . '</span>';
                    }
                })


                ->addColumn('action', function ($row) use ($privileges) {
                    $actionBtn = '<div class="flex items-center justify-center gap-2">';
                    if (in_array('Edit Requisition', $privileges)) {
                        $actionBtn .= '<a href="' . route('requisitions.edit', ['requisition' => $row->id]) . '" class="flex items-center justify-center w-10 h-10 text-white bg-blue-600 rounded-md hover:bg-blue-900 edit"><i class="bx bx-edit-alt"></i></a>';
                    }
                    if (in_array('Delete Requisition', $privileges)) {
                        $actionBtn .= '  <button onclick="openDeleteModal(' . $row->id . ')" class="flex items-center justify-center w-10 h-10 text-white bg-red-600 rounded-md">
                        <i class="text-xl bx bx-trash-alt"></i>
                    </button>';
                    }
                    $actionBtn .= '</div>';
                    return $actionBtn;
                })
                ->rawColumns(['action', 'acceptbutton'])
                ->make(true);
        }
        $employees = Employee::where('status', 3)->get();
        $agencyQuery = Agency::query();
        if (!empty($agencyIds)) {
            $agencyQuery->whereIn('id', $agencyIds);
        }
        $agencies = $agencyQuery->get();


        $data = Requisition::with('client.agency', 'department')->orderBy('created_at', 'desc')->get();
        $client = Client::with('user', 'agency')->get();
        $client1 = Client::with('user', 'agency')->get();
        $departments = Department::get();

        $stores = Stores::all();


        return view('Admin.Requisition.index', compact('data', 'stores', 'agencies', 'client', 'client1', 'departments', 'employees'));
    }




    public function statusupdate(Request $request)
    {
        $id = $request->input('id');
        $requisition = Requisition::find($id);
        $status = $request->input('status');
        $action = MrfAction::create([
            'mrf_id' => $id,
            'user_id' => auth()->id(),
            'action' => $status,
        ]);


        $req = Requisition::find($id);
        $req->status = $status;
        $req->update();


        return redirect()->back()->with('success', 'Requisition Approved.');
    }
    public function edit(Requisition $requisition)
    {
        $requisition->load('client', 'department', 'mrfapplicant');
        $agencies = Agency::all();
        $companies = Company::all();
        $clients = Client::all();
        $departments = Department::all();
        $stores = Stores::all();
        $offices = Offices::all();

        $employees = Employee::with('channelclient')  // Eager load related channelclient
            ->where(function ($query) {
                $query->where('status', 3)  // Employees with status 3 in the Employee table
                    ->orWhereDoesntHave('channelclient');  // OR Employees with no related channelclient
            })
            ->get();

        $mrf = MrfAction::with('user', 'requisition')->where('mrf_id', $requisition->id)->get();
        $details = MrfMoreDetail::where('mrf_id', $requisition->id)->get();


        return view('Admin.Requisition.edit', compact('stores', 'offices', 'requisition', 'details', 'agencies', 'clients', 'companies', 'employees', 'mrf', 'departments'));
    }





    public function create()
    {
        $clients = Client::all();
        $departments = Department::all();

        return view('requisitions.create', compact('clients', 'departments'));
    }
    public function getCompaniesByAgency($agencyId)
    {

        $user = Auth::user();
        $agencyIds = $user->viewable_agency_ids;
        $query = Client::where('agency_id', $agencyId);


        if (!empty($agencyIds)) {

            $query->whereIn('agency_id', $agencyIds);


            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('company_id')
                ->unique();

            if ($restrictedCompanyIds->isNotEmpty()) {
                $query->whereIn('id', $restrictedCompanyIds);
            }

        }
        $companies = $query->get();

        return response()->json($companies);
    }

    public function getClientsByCompany($companyId)
    {
        $user = Auth::user();
        $agencyIds = $user->viewable_agency_ids;

        $query = Department::with('client.agency')->where('client_id', $companyId);



        if (!empty($agencyIds)) {

            $query->whereHas('client.agency', function ($q) use ($agencyIds) {
                $q->whereIn('id', $agencyIds);
            });

            $assignIds = Assign::where('user_id', $user->id)
                ->whereIn('agency_id', $agencyIds)
                ->pluck('id');

            $restrictedCompanyIds = AssignCompany::whereIn('assign_id', $assignIds)
                ->pluck('company_id')
                ->unique();

            if ($restrictedCompanyIds->isNotEmpty()) {
                $query->whereIn('client_id', $restrictedCompanyIds);
            }

            $restrictedDepartmentIds = AssignCompanyDepartment::whereIn(
                'assigncompany_id',
                AssignCompany::whereIn('assign_id', $assignIds)->pluck('id')
            )->pluck('department_id')->unique();

            if ($restrictedDepartmentIds->isNotEmpty()) {
                $query->whereIn('id', $restrictedDepartmentIds);
            }

        }
        $clients = $query->get();

        return response()->json($clients);
    }


    public function getDepartmentsByClient($clientId)
    {
        $departments = Department::where('client_id', $clientId)->get();
        return response()->json($departments);
    }
    public function store(Request $request)
    {
        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'department_id' => 'required|exists:department,id',
            'designation' => 'nullable|string|max:255',
            'date' => 'nullable|date',
            'job_title' => 'nullable|string|max:255',
            'vacant_no' => 'nullable|string|max:255',
            'rec_type' => 'nullable|string|max:255',
            'budget_salary' => 'nullable|string|max:255',
            'filled_date' => 'nullable|date',
            'project_name' => 'nullable|string|max:255',
            'replacement' => 'nullable|string|max:255',
            'reasons' => 'nullable|string|max:255',
            'vacant_form' => 'nullable|string|max:255',
            'employee_ids' => 'nullable|array',
            'employee_ids.*' => 'exists:employees,id',
            'status' => 'required|string|max:255',
            'store_id' => 'nullable|string|max:255',
            'gender' => 'nullable|string|max:255',
            'per_store' => 'nullable|array',
            'per_store.*' => 'nullable|string',
            'frequency' => 'nullable|array',
            'frequency.*' => 'nullable|string',
        ]);

        try {
            DB::transaction(function () use ($request) {
                $requisition = Requisition::create($request->only([
                    'client_id',
                    'department_id',
                    'store_id',
                    'designation',
                    'date',
                    'job_title',
                    'vacant_no',
                    'rec_type',
                    'budget_salary',
                    'filled_date',
                    'project_name',
                    'replacement',
                    'reasons',
                    'gender',
                    'vacant_form',
                ]));

                if ($request->has('employee_ids') && is_array($request->employee_ids)) {
                    $data = [];
                    foreach ($request->employee_ids as $employee_id) {
                        $data[] = [
                            'requisition_id' => $requisition->id,
                            'employee_id' => $employee_id,
                            'status' => $request->status ?? 'default_status',
                            'created_at' => now(),
                            'updated_at' => now(),
                        ];
                    }
                    MrfApplicant::insert($data);
                }

                foreach ($request->per_store ?? [] as $index => $perStore) {
                    if (!empty($perStore) && !empty($request->frequency[$index])) {
                        MrfMoreDetail::create([
                            'mrf_id' => $requisition->id,
                            'per_store' => $perStore,
                            'frequency' => $request->frequency[$index],
                        ]);
                    }
                }
            });

            return response()->json([
                'message' => 'Requisition and MRF Applicants created successfully.'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to save requisition.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }



    public function storeapplicant(Request $request)
    {
        $request->validate([
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:employees,id',
            'status' => 'required|string|max:255',
        ]);

        $requisitionId = $request->input('requisition_id');

        DB::transaction(function () use ($request, $requisitionId) {
            foreach ($request->employee_ids as $employeeId) {
                MrfApplicant::create([
                    'requisition_id' => $requisitionId,
                    'employee_id' => $employeeId,
                    'status' => $request->status,
                ]);
            }
        });

        return redirect()->back()
            ->with('success', 'Applicants added successfully.');
    }
    public function updateapplicant(Request $request)
    {
        // Validate input data (only 'update_status' is required now)
        $request->validate([
            'update_status' => 'required|array',
            'update_status.*' => 'required|string|max:255|in:open,approved,for_approval,meet_greet_intro,process_requirement,agency_orientation,client_orientation,deploy,disapproved,cancel',
        ]);

        foreach ($request->input('update_status') as $id => $status) {
            $applicant = MrfApplicant::find($id);

            if ($applicant) {
                $applicant->status = $status;
                $applicant->save();

                $date = $request->input("update_date.$id");
                $remarks = $request->input("remarks.$id");


                ApplicantLog::create([
                    'applicant_id' => $id,
                    'user_id' => Auth::id(),
                    'date' => $date ?? null, // Use provided date or current timestamp if none
                    'action' => 'Status updated to: ' . ucfirst($status),
                    'remarks' => $remarks ?? 'No remarks',
                ]);
            } else {
                return redirect()->back()->with('error', 'One or more applicants were not found.');
            }
        }

        // Redirect back with success message
        return redirect()->back()->with('success', 'Applicants updated successfully.');
    }




    public function show(Requisition $requisition)
    {
        return view('requisitions.show', compact('requisition'));
    }


    public function update(Request $request, Requisition $requisition)
    {
        $request->validate([
            'client_id' => 'nullable|exists:clients,id',
            'department_id' => 'nullable|exists:department,id',
            'designation' => 'nullable|string|max:255',
            'date' => 'nullable|date',
            'job_title' => 'nullable|string|max:255',
            'vacant_no' => 'nullable|string|max:255',
            'rec_type' => 'nullable|string|max:255',
            'budget_salary' => 'nullable|string|max:255',
            'filled_date' => 'nullable|date',
            'project_name' => 'nullable|string|max:255',
            'replacement' => 'nullable|string|max:255',
            'reasons' => 'nullable|string|max:255',
            'vacant_form' => 'nullable|string|max:255',

            'status' => 'nullable|string|max:255',
        ]);

        DB::transaction(function () use ($request, $requisition) {
            // Update the requisition
            $requisition->update($request->only([
                'client_id',
                'department_id',
                'designation',
                'date',
                'job_title',
                'vacant_no',
                'rec_type',
                'budget_salary',
                'filled_date',
                'project_name',
                'replacement',
                'reasons',
                'vacant_form',
                'status'
            ]));
        });

        return redirect()->back()->with('success', 'Requisition updated successfully.');
    }

    public function destroy($id)
    {
        $requisition = Requisition::find($id);
        $requisition->delete();
        return response()->json(['success' => 'Requisition and MRF Applicants created successfully.']);
    }
}
