<?php

namespace App\Http\Controllers;

use Log;
use App\Models\TrxCode;
use App\Models\TrxChannel;
use App\Models\TrxGroupcode;

use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\DB;

class TrxCodeController extends Controller
{
    public function index(Request $request)
    {
        $codes = TrxCode::with('groupCode')->get();


        if ($request->ajax()) {

            $data = TrxCode::with('groupCode', 'options')->get();

            // DataTables setup
            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    // Pass necessary data as data- attributes
                    return '<button class="btn btn-info bg-blue-500 text-white p-2 rounded-md" 
                        onclick="openModal(this)" 
                        data-id="' . $row->id . '"
                        data-group-code-id="' . $row->group_code_id . '"
                        data-description="' . $row->description . '"
                        data-abbreviation="' . $row->abbreviation . '"
                        data-percentage="' . $row->percentage . '"
                        data-entry-order="' . $row->entry_order . '"
                        data-sub-category="' . $row->sub_category . '"
                        data-options="' . json_encode($row->options->pluck('id')->toArray()) . '">
                    Edit
                </button>';
                })
                ->rawColumns(['action'])
                ->make(true);
        }
    }

    public function create()
    {
        $groupcodes = TrxGroupcode::all();
        return view('trx_codes.create', compact('groupcodes'));
    }

    public function store(Request $request)
    {
        // Validate the incoming request
        $request->validate([
            'group_code_id' => 'required|exists:trx_groupcodes,id',
            'description' => 'nullable|string|max:255',
            'abbreviation' => 'nullable|string|max:10',
            'percentage' => 'nullable|numeric',
            'entry_order' => 'nullable|integer',
            'sub_category' => 'nullable|string|max:255',
            'option_ids' => 'nullable|array', // Array of selected options
            'option_ids.*' => 'nullable|exists:trx_options,id', // Validate each selected option ID
        ]);

        // Create the new TrxCode
        $trxCode = TrxCode::create([
            'group_code_id' => $request->group_code_id,
            'description' => $request->description,
            'abbreviation' => $request->abbreviation,
            'percentage' => $request->percentage,
            'entry_order' => $request->entry_order,
            'sub_category' => $request->sub_category,
        ]);

        // If option_ids are selected, associate them with the created TrxCode
        if ($request->has('option_ids')) {
            foreach ($request->option_ids as $optionId) {
                TrxChannel::create([
                    'code_id' => $trxCode->id, // Associate with the created TrxCode
                    'option_id' => $optionId, // Use the selected option ID
                ]);
            }
        }

        return redirect()->back()->with('success', 'TrxCode and associated options created successfully.');
    }


    public function show(TrxCode $trxCode)
    {
        return view('trx_codes.show', compact('trxCode'));
    }

    public function edit(TrxCode $trxCode)
    {
        $groupcodes = TrxGroupcode::all();
        return view('trx_codes.edit', compact('trxCode', 'groupcodes'));
    }


    public function update(Request $request, $id)
    {
        // Validate the incoming request
        $validatedData = $request->validate([
            'description' => 'nullable|string|max:255',
            'abbreviation' => 'nullable|string|max:10',
            'percentage' => 'nullable|numeric|min:0',
            'entry_order' => 'nullable|integer|min:1',
            'sub_category' => 'nullable|string|max:255',
            'option_ids' => 'nullable|array', // Array of selected options
            'option_ids.*' => 'nullable|exists:trx_options,id', // Validate each selected option ID
        ]);

        // Use a database transaction to ensure data integrity
        DB::beginTransaction();

        try {
            // Find the TrxCode by ID or fail
            $trxCode = TrxCode::findOrFail($id);

            // Update the TrxCode model with the validated data
            $trxCode->update([
                'description' => $validatedData['description'] ?? null,
                'abbreviation' => $validatedData['abbreviation'] ?? null,
                'percentage' => $validatedData['percentage'] ?? null,
                'entry_order' => $validatedData['entry_order'] ?? null,
                'sub_category' => $validatedData['sub_category'] ?? null,
            ]);

            // Check if option_ids is provided in the request
            if (isset($validatedData['option_ids'])) {
                // Get the current option IDs associated with this TrxCode
                $currentOptionIds = TrxChannel::where('code_id', $trxCode->id)->pluck('option_id')->toArray();

                // Determine which options need to be added or removed
                $optionsToAdd = array_diff($validatedData['option_ids'], $currentOptionIds);
                $optionsToRemove = array_diff($currentOptionIds, $validatedData['option_ids']);

                // Remove options that are no longer selected
                if (!empty($optionsToRemove)) {
                    TrxChannel::where('code_id', $trxCode->id)
                        ->whereIn('option_id', $optionsToRemove)
                        ->delete();
                }

                // Add the new options
                if (!empty($optionsToAdd)) {
                    $optionsData = array_map(function ($optionId) use ($trxCode) {
                        return [
                            'code_id' => $trxCode->id,
                            'option_id' => $optionId,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ];
                    }, $optionsToAdd);

                    TrxChannel::insert($optionsData);
                }
            }

            // Commit the transaction if everything is successful
            DB::commit();

            // Redirect back with success message
            return redirect()->back()->with('success', 'TrxCode and associated options updated successfully.');
        } catch (\Exception $e) {
            // Rollback the transaction in case of an error
            DB::rollBack();

            // Optionally log the error for debugging


            // Redirect back with error message
            return redirect()->back()->with('error', 'An error occurred while updating the TrxCode.');
        }
    }


    public function destroy(TrxCode $trxCode)
    {
        $trxCode->delete();
        return redirect()->route('trx-codes.index')->with('success', 'Code deleted successfully.');
    }
}
