<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use App\Models\Agency;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Yajra\DataTables\DataTables;
use App\Models\Department;
use App\Models\Employee;

class UserController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            // Fetch users excluding the ones with role_id = 2
            $users = User::with('role')->where('role_id', '!=', '2');

            return DataTables::of($users)
                ->addColumn('role', function ($user) {
                    // Get role name for each user
                    return $user->role ? $user->role->role_name : 'No role assigned';
                })
                ->addColumn('name', function ($user) {
                    $name = '';

                    if ($user->role_id == 8) {
                        $client = Department::where('user_id', $user->id)->first();
                        if ($client) {
                            $name = $client->department_name;
                        }
                    } elseif ($user->role_id == 9) {
                        $employee = Employee::where('user_id', $user->id)->first();
                        if ($employee) {
                            $name = $employee->first_name . ' ' . $employee->last_name;
                        }
                    } else {
                        $name = $user->name;
                    }

                    return $name;
                })
                ->addColumn('action', function ($user) {
    // Add action buttons (edit, delete)
    $actionBtn = '
    <div class="flex items-center justify-center gap-2">
        <div class="bg-blue-600 rounded-md">
        <a href="' . route('assigns.get', $user->id) . '" class="flex items-center justify-center w-10 h-8 text-white ">
            <i class="text-xl bx bx-street-view"></i>
        </a>
        </div>
        <button onclick="openModal(\'updateModal\', ' . $user->id . ')" class="flex items-center justify-center w-10 h-8 text-white bg-blue-600 rounded-md">
            <i class="text-xl bx bx-edit-alt"></i>
        </button>';
        
    if ($user->id != 1) {
        $actionBtn .= '
        <button onclick="openModal(\'deleteModal\', ' . $user->id . ')" class="flex items-center justify-center w-10 h-8 text-white bg-red-600 rounded-md">
            <i class="text-xl bx bx-trash-alt"></i>
        </button>';
    }

    $actionBtn .= '</div>';
    return $actionBtn;
})
                ->rawColumns(['action'])
                ->make(true);
        }

        // If it's not an AJAX request, fetch data and return the normal view
        $roles = Role::all();  // Fetch all roles (to populate the select input in the modal)
        return view('Admin.User.index', compact('roles'));
    }



    public function create()
    {
        return view('users.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|confirmed|min:8',
            'role_id' => 'required|exists:roles,id',
        ]);

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role_id' => $request->role_id,
        ]);

        return response()->json(['success' => true, 'message' => 'User created successfully.']);
    }

    public function show(User $user)
    {
        return view('users.show', compact('user'));
    }

    public function edit($id)
    {
        $user = User::findOrFail($id);
        $roles = Role::all(); // Get all roles

        return response()->json([
            'user' => $user,
            'roles' => $roles,
            'current_role_id' => $user->role_id // Include the current role ID
        ]);
    }


    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        // Validate input
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'role_id' => 'required|exists:roles,id',
            'password' => 'nullable|min:3|', // password is optional
        ]);

        // Update user data (excluding password unless it's provided)
        $data = $request->only('name', 'email', 'role_id');

        if ($request->filled('password')) {
            // If password is provided, hash it
            $data['password'] = bcrypt($request->password);
        }

        // Update the user with new data
        $user->update($data);

        // Optionally return the updated user as JSON to update the frontend
        return response()->json(['success' => true]);
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();

        // Return a response to indicate the action is successful (it might not be needed but good for debugging)
        return response()->json(['success' => true]);
    }
}
