<?php


namespace App\Imports;

use App\Models\Attendance;
use App\Models\Employee;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Illuminate\Support\Carbon;

class AttendanceImport implements ToModel, WithHeadingRow
{
    public function model(array $row)
    {
        // Trim spaces from the date value if necessary
        $dateString = trim($row['date']);

        // Handle Excel serial date (convert from serial number to date)
        if (is_numeric($dateString)) {
            // Convert Excel serial date to DateTime
            $date = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($dateString);
        } else {
            // Handle non-serial date
            try {
                $date = Carbon::createFromFormat('m/d/Y', $dateString);
            } catch (\Exception $e) {
                $date = null;
            }
        }

        // Ensure the employee exists
        $employee = Employee::where('last_name', 'like', '%' . trim($row['last_name']) . '%')
            ->where('first_name', 'like', '%' . trim($row['first_name']) . '%')
            ->when(!empty($row['middle_name']), function ($query) use ($row) {
                // Add middle name condition only if it's provided
                $query->where('middle_name', 'like', '%' . trim($row['middle_name']) . '%');
            })
            ->first();

        // Time handling: Convert time_in and time_out fields to 24-hour format with seconds
        $timeIn = $this->parseTime(trim($row['time_in']));
        $timeOut = $this->parseTime(trim($row['time_out']));

        return new Attendance([
            'employee_id' => $employee->id,
            'date' => $date ? $date->format('Y-m-d') : null, // Ensure the date is formatted correctly
            'time_in' => $timeIn->format('H:i:s'), // Convert time_in to 24-hour format with seconds
            'time_out' => $timeOut->format('H:i:s'), // Convert time_out to 24-hour format with seconds
            'notes' => isset($row['notes']) ? $row['notes'] : null,
            'status' => isset($row['status']) ? $row['status'] : null,
        ]);
    }

    // Helper function to parse time with both 12-hour (h:i A) and 24-hour (H:i) formats
    private function parseTime($time)
    {
        try {
            // Check if time is an Excel serial value (a number)
            if (is_numeric($time)) {
                // If so, convert the serial value to a DateTime object
                return \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($time);
            }

            // If not, proceed with the usual parsing logic
            try {
                // First, try the 12-hour format with seconds (h:i:s A)
                return Carbon::createFromFormat('h:i:s A', $time);
            } catch (\Exception $e) {
                try {
                    // If it fails, try the 12-hour format without seconds (h:i A)
                    return Carbon::createFromFormat('h:i A', $time);
                } catch (\Exception $e) {
                    try {
                        // If it fails again, try the 24-hour format without AM/PM (H:i)
                        return Carbon::createFromFormat('H:i', $time);
                    } catch (\Exception $e) {
                        // If all formats fail, log the error and return a default time

                        return Carbon::now(); // Return the current time as fallback
                    }
                }
            }
        } catch (\Exception $e) {

            return Carbon::now(); // Return the current time as fallback if there's an issue
        }
    }
}
