<?php

namespace App\Imports;

use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Carbon\Carbon;
use App\Models\Employee;
use App\Models\EmployeeClientChannel;
use App\Models\Attendance;
use Illuminate\Support\Facades\DB;
use Exception;

class AttendanceImport3 implements ToCollection
{
    const BATCH_SIZE = 10000;

    public $failedRows = []; // Array to store failed rows
    public $departmentId;
    /**
     * @param Collection $rows
     */
    public function __construct($departmentId)
    {
        $this->departmentId = $departmentId; // Store the department ID
    }
    public function collection(Collection $rows)
    {
        $header = ['ID', 'DateTime', 'Column 2'];

        if ($rows->first()->values()->toArray() === $header) {
            $rows->shift(); // Remove header row
        }

        $lineNumber = 1; // Start from line 2 since the header is skipped
        $attendanceData = [];
        $results = [];

        $groupedData = $rows->groupBy(function ($row) {
            $employeeCode = $row[0];
            $date = $this->parseDateTime($row[1]);
            return $date ? $employeeCode . '-' . $date->format('Y-m-d') : null;
        });

        foreach ($groupedData as $key => $entries) {
            if (!$key) {
                foreach ($entries as $row) {
                    $results[] = [
                        'line' => $lineNumber++,
                        'ID' => $row[0],
                        'DateTime' => $row[1],
                        'Column 2' => $row[2],
                        'error' => 'Invalid DateTime format. Please ensure the DateTime is valid.',
                    ];
                }
                continue;
            }

            $employeeCode = $entries->first()[0];
            $date = $this->parseDateTime($entries->first()[1])->format('Y-m-d');

            $employeeChannels = EmployeeClientChannel::where('status', '1')
                ->where('dtr_id', $employeeCode)
                ->where('department_id', $this->departmentId)
                ->get();

            if ($employeeChannels->isEmpty()) {
                foreach ($entries as $row) {
                    $results[] = [
                        'line' => $lineNumber++,
                        'ID' => $row[0],
                        'DateTime' => $row[1],
                        'Column 2' => $row[2],
                        'error' => "No active employee found for ID: $employeeCode in department {$this->departmentId}.",
                    ];
                }
                continue;
            }

            $timeIn = null;
            $timeOut = null;
            $outDate = null;

            foreach ($entries as $entry) {
                $timestamp = $this->parseDateTime($entry[1]);

                if ($entry[2] == 0 && !$timeIn) {
                    $timeIn = $timestamp;
                }

                if ($entry[2] == 1) {
                    $timeOut = $timestamp;

                    // If time-out is earlier than time-in, assume it’s from the next day
                    if ($timeIn && $timeOut->lessThan($timeIn)) {
                        $outDate = $timeIn->copy()->addDay()->format('Y-m-d');
                    } else {
                        $outDate = $timeOut->format('Y-m-d');
                    }
                }
            }

            foreach ($employeeChannels as $channel) {
                $exists = DB::table('attendances')
                    ->where('employee_id', $channel->employee_id)
                    ->where('date', $date)
                    ->where('department_id', $this->departmentId)
                    ->exists();

                if ($exists) {
                    foreach ($entries as $row) {
                        $results[] = [
                            'line' => $lineNumber++,
                            'ID' => $row[0],
                            'DateTime' => $row[1],
                            'Column 2' => $row[2],
                            'error' => "Attendance already exists for employee ID: {$channel->employee_id} on {$date}.",
                        ];
                    }
                    continue;
                }

                $attendanceData[] = [
                    'employee_id' => $channel->employee_id,
                    'department_id' => $this->departmentId,
                    'date' => $date,
                    'time_in' => $timeIn,
                    'time_out' => $timeOut,
                    'out_date' => $outDate,
                    'notes' => 'Imported data',
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                foreach ($entries as $row) {
                    $results[] = [
                        'line' => $lineNumber++,
                        'ID' => $row[0],
                        'DateTime' => $row[1],
                        'Column 2' => $row[2],
                        'error' => 'Inserted successfully.',
                    ];
                }
            }
        }

        if (!empty($attendanceData)) {
            $this->insertAttendanceBatch($attendanceData);
        }

        // Store results
        $this->failedRows = $results;
    }


    // Helper method to insert the batch of data
    private function insertAttendanceBatch($attendanceData)
    {
        DB::table('attendances')->insert($attendanceData);
    }

    // Parse the DateTime from the various formats
    private function parseDateTime($dateTime)
    {
        $dateTime = trim($dateTime);
        if (is_numeric($dateTime)) {
            return Carbon::createFromFormat('Y-m-d H:i', gmdate('Y-m-d H:i', ($dateTime - 25569) * 86400));
        }

        $formats = [
            'n/j/Y H:i',
            'm/d/Y H:i',
            'Y-m-d H:i',
            'n/j/Y g:i',
            'm/d/Y g:i',
            'Y-m-d g:i',
            'Y-m-d H:i:s',
            'm/d/Y H:i:s',
            'n/j/Y H:i:s',
            'n/j/Y g:i:s',
            'm/d/Y H:i A',
            'n/j/Y g:i A',
            'Y-m-d g:i A',
            'Y-m-d h:i A',
            'm/d/Y h:i A',
            'n/j/Y h:i A',
            'Y-m-d h:i A',
            'Y/m/d H:i',
            'Y.m.d H:i',
            'd/m/Y H:i',
            'd.m.Y H:i',
            'd/m/Y g:i A',
            'd.m.Y g:i A',
            'd/m/Y h:i A',
            'd.m.Y h:i A',
            'Y-m-d\TH:i:s\Z',
            'D, d M Y H:i:s O',
            'D, d M Y H:i:s T',
            'Y-m-d\TH:i:s',
            'Y/m/d H:i:s',
            'Y-m-d H:i:s.u',
        ];

        foreach ($formats as $format) {
            try {
                return Carbon::createFromFormat($format, $dateTime);
            } catch (Exception $e) {
                continue;
            }
        }

        return null;
    }
}
