<?php

namespace App\Imports;

use App\Models\User;
use App\Models\Employee;
use App\Models\Employeerate;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Illuminate\Contracts\Queue\ShouldQueue;
use Carbon\Carbon;

class EmployeeImport implements ToModel, WithHeadingRow, WithChunkReading
{
    protected $importedRows = [];
    protected $skippedRows = [];
    protected $currentRow = 1;

    public function model(array $row)
    {
        $rowNumber = $this->currentRow++;

        // Skip empty rows
        if (empty(array_filter($row))) {
            $this->skippedRows[] = [
                'row' => $rowNumber,
                'reason' => 'Empty row',
                'data' => $row,
            ];
            return null;
        }

        // Parse employee name
        $nameParts = explode(',', $row['employee_name']);
        $lastName = trim($nameParts[0] ?? '');
        $firstAndMiddle = isset($nameParts[1]) ? explode(' ', trim($nameParts[1])) : [];
        $firstName = $firstAndMiddle[0] ?? '';
        $middleName = $firstAndMiddle[1] ?? '';

        if (!$firstName || !$lastName) {
            $this->skippedRows[] = [
                'row' => $rowNumber,
                'reason' => 'Missing first or last name',
                'data' => $row,
            ];
            return null;
        }

        // Check duplicates
        $existingEmployee = Employee::where('first_name', $firstName)
            ->where('middle_name', $middleName)
            ->where('last_name', $lastName)
            ->first();

        if ($existingEmployee) {
            $this->skippedRows[] = [
                'row' => $rowNumber,
                'name' => "$firstName $middleName $lastName",
                'mobile' => $row['mobile_no'] ?? '',
                'reason' => 'Already exists',
                'data' => $row,
            ];
            return null;
        }

        // Parse dates safely
        $dob = $this->parseDate($row['date_of_birth'] ?? null);
        $dateHired = $this->parseDate($row['date_hired'] ?? null);

// Generate base email
$emailBase = strtolower($firstName . $lastName . $middleName . '@gmail.com');
$email = $emailBase;
$counter = 1;

// Ensure unique email across DB
while (User::where('email', $email)->exists()) {
    $email = strtolower($firstName . $lastName . $middleName . $counter . '@gmail.com');
    $counter++;
}

// Create new user (always unique)
$user = User::create([
    'name' => "$firstName $middleName $lastName",
    'email' => $email,
    'password' => Hash::make('123'),
    'role_id' => 9,
]);

// Create employee with correct user_id
$employee = Employee::create([
    'first_name'        => $firstName,
    'middle_name'       => $middleName,
    'last_name'         => $lastName,
    'date_of_birth'     => $dob,
    'sex'               => $row['sex'] ?? null,
    'age'               => $row['age'] ?? null,
    'civil_status'      => [
        'S' => 'single',
        'M' => 'married',
        'H' => 'head_of_family',
        'W' => 'widow',
    ][strtoupper($row['civil_status'] ?? '')] ?? null,
    'mobile_number'     => $row['mobile_no'] ?? null,
    'present_address'   => $row['address'] ?? null,
    'provincial_address'=> $row['provincial_address'] ?? null,
    'sss_no'            => $row['sss_no'] ?? null,
    'pagibig_no'        => $row['hdmf_no'] ?? null,
    'philhealth_no'     => $row['phic_no'] ?? null,
    'tin_no'            => $row['tin'] ?? null,
    'bank_number'       => $row['account_no'] ?? null,
    'bank_name'         => $row['payment_remittance'] ?? null,
    'user_id'           => $user->id,   // ✅ now always unique
    'status'            => 1,
    'position_applied'  => $row['manpower_type'] ?? null,
]);

        // Create salary rate
        $this->createEmployeerate($employee->id, $row, $dateHired);

        // Log imported row
        $this->importedRows[] = [
            'row' => $rowNumber,
            'name' => "$firstName $middleName $lastName",
            'email' => $email,
            'dob' => $dob,
            'mobile' => $row['mobile_no'],
            'address' => $row['address'] ?? ''
        ];

        return null;
    }

    private function createEmployeerate($employeeId, $row, $dateHired)
    {
        Employeerate::firstOrCreate(
            [
                'employee_id' => $employeeId,
                'date' => $dateHired
            ],
            [
                'status' => 1,
                'monthly_salary' => $row['monthly_rate'] ?? 0,
                'daily_rate' => $row['daily_rate'],
                'cola_1' => 0,
                'cola_2' => 0,
                'allowance' => 0,
            ]
        );
    }

    private function parseDate($value)
    {
        if (empty($value)) {
            return null;
        }

        // If it's an Excel numeric date serial
        if (is_numeric($value)) {
            try {
                return Carbon::instance(\PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($value))
                    ->format('Y-m-d');
            } catch (\Exception $e) {
                return null;
            }
        }

        // Try d/m/Y first
        try {
            return Carbon::createFromFormat('d/m/Y', trim($value))->format('Y-m-d');
        } catch (\Exception $e) {
            // If that fails, try m/d/Y (Excel default in some locales)
            try {
                return Carbon::createFromFormat('m/d/Y', trim($value))->format('Y-m-d');
            } catch (\Exception $e) {
                return null; // Invalid date format
            }
        }
    }

    public function chunkSize(): int
    {
        return 500;
    }

    public function getImportedRows(): array
    {
        return $this->importedRows;
    }

    public function getSkippedRows(): array
    {
        return $this->skippedRows;
    }
}
