<?php

namespace App\Imports;

use App\Models\Ptsr;
use App\Models\PtsrTrx;
use App\Models\Employee;
use App\Models\EmployeeClientChannel;
use App\Models\TrxCode;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Illuminate\Support\Facades\Log;

class PtsrImport implements ToModel, WithHeadingRow
{
    protected $startDate;
    protected $endDate;
    protected $department_id;
    public $failedRows = [];

    public function __construct($startDate, $endDate, $department_id)
    {
        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->department_id = $department_id;
    }

    public function model(array $row)
    {
        $fullName = trim($row['employee_name'] ?? '');
        if (!$fullName) return null; // Skip empty rows

        // Remove any leading numbers or punctuation (e.g., "2. ", "1. ")
        $fullName = preg_replace('/^\d+(\.)?\s+/', '', $fullName);

        // Replace non-breaking spaces with regular spaces
        $fullName = str_replace("\xC2\xA0", ' ', $fullName);

        // Normalize spaces to single spaces
        $fullName = preg_replace('/\s+/', ' ', trim($fullName));

        // Continue with the original logic
        $nameParts = explode(',', $fullName, 2);

        if (count($nameParts) < 2) {
            Log::error("Invalid name format: {$fullName}");
            return null;
        }

        $lastName = trim($nameParts[0]); // Extract last name
        $firstMiddle = trim($nameParts[1]); // Extract first and middle names

        // Extract first name and middle name(s)
        $firstMiddleParts = explode(' ', $firstMiddle);
        $firstName = array_shift($firstMiddleParts); // Take the first word as the first name
        $middleName = !empty($firstMiddleParts) ? strtoupper(implode(' ', $firstMiddleParts)) : null; // Join the rest as the middle name

        // Debugging - Log extracted names
        Log::info("Extracted Name - Last: {$lastName}, First: {$firstName}, Middle: {$middleName}");


        // Find Employee in Database
        $employee = Employee::where('last_name', 'like', "%{$lastName}%")
            ->where('first_name', 'like', "%{$firstName}%")
            ->when($middleName, function ($query) use ($middleName) {
                return $query->where('middle_name', 'like', "%{$middleName}%");
            })
            ->first();



        // Row status tracking
        $rowStatus = [
            'row_data'  => $row,
            'status'    => 'success',
            'errors'    => [],
            'trx_codes' => []
        ];

        if (!$employee) {
            $errorMessage = "Employee not found: {$lastName}, {$firstName}" . ($middleName ? " {$middleName}" : '');
            Log::error($errorMessage);
            $rowStatus['status'] = 'failed';
            $rowStatus['errors'][] = $errorMessage;
            $this->failedRows[] = $rowStatus;
            return null;
        }

        // Define allowed fields and map them dynamically
        $mapping = [
            'under_time'                => 'under_time',
            'total_absent'              => 'total',
            'adj'                       => 'adj',
            'ot'                        => 'ot',
            'night_prem'                => 'night',
            'special_premium'           => 'special_premium',
            'legal_holiday'             => 'legal_holiday',
            'duty_restday_premium_hour' => 'duty',
            'no_of_dys_wrk'             => 'number_of_days'
        ];

        $employeeChannels = EmployeeClientChannel::where('status', '1')
            ->where('employee_id', $employee->id)
            ->where('department_id', $this->department_id)
            ->exists();

        if (!$employeeChannels) {
            $errorMessage = "Employee {$lastName}, {$firstName} is not assigned to department ID: {$this->department_id}";
            Log::error($errorMessage);

            $rowStatus['status'] = 'failed';
            $rowStatus['errors'][] = $errorMessage;
            $this->failedRows[] = $rowStatus;

            return null; // ❌ Skip processing if department does not match
        }


        // Dynamically build the PTSR data array
        $ptsrData = [
            'employee_id' => $employee->id,
            'department_id' => $this->department_id,
            'start_date'  => $this->startDate,
            'end_date'    => $this->endDate,
            'status'      => 'ptsr',
        ];

        // Map the correct headers dynamically
        foreach ($mapping as $header => $column) {
            // Ensure the header exists in the row
            if (isset($row[$header])) {
                // Convert encoding and trim whitespace
                $value = trim(mb_convert_encoding($row[$header], 'UTF-8', 'auto'));

                // Clean non-breaking spaces (for Google Sheets)
                $value = str_replace("\xC2\xA0", ' ', $value);

                // Remove all non-numeric characters except for the decimal point
                $cleanValue = preg_replace('/[^0-9.]/', '', $value);

                // If the cleaned value is numeric, convert it to a float
                if (is_numeric($cleanValue)) {
                    $ptsrData[$column] = (float) $cleanValue;
                } else {
                    // If not numeric, log a warning and set the value to 0 (default)
                    Log::warning("Invalid numeric value for '{$header}': '{$value}', defaulting to 0");
                    $ptsrData[$column] = 0;
                }

                // Debugging - log the raw and cleaned values
                Log::info("Processed '{$header}': Raw='{$value}', Clean='{$cleanValue}', Final='{$ptsrData[$column]}'");
            } else {
                // If header does not exist in the row, log an error and set the value to 0
                Log::error("Header '{$header}' not found in row!");
                $ptsrData[$column] = 0;
            }
        }
        // Save PTSR record
        $ptsr = Ptsr::create($ptsrData);

        // Save Additional Trx Codes dynamically
        foreach ($row as $header => $value) {
            // Skip empty values
            if (empty($value)) continue;

            // Clean and process the value
            $value = trim(mb_convert_encoding($value, 'UTF-8', 'auto'));

            // Clean non-breaking spaces (for Google Sheets)
            $value = str_replace("\xC2\xA0", ' ', $value);

            // Remove all non-numeric characters except for the decimal point
            $cleanValue = preg_replace('/[^0-9.]/', '', $value);

            // Check if the value is numeric
            if (is_numeric($cleanValue)) {
                // If numeric, we proceed as usual for both mapped and non-mapped fields
                $ptsrData[$header] = (float) $cleanValue;
            } else {
                // If not numeric, log a warning
                Log::warning("Invalid numeric value for '{$header}': '{$value}', defaulting to 0");

                // Set the value to 0 if it's not a valid number
                $ptsrData[$header] = 0;
            }

            // Check if this field is also a TrxCode
            $normalizedHeader = strtolower(trim(preg_replace('/\s+/', ' ', $header)));

            // Try to find matching TrxCode for the header
            $trxCode = TrxCode::whereRaw("LOWER(TRIM(REPLACE(description, '  ', ' '))) LIKE ?", ["%{$normalizedHeader}%"])->first();

            if ($trxCode) {
                // Create a transaction record if TrxCode is found
                PtsrTrx::create([
                    'ptsr_id' => $ptsr->id,
                    'code_id' => $trxCode->id,
                    'amount'  => (float) $value,
                ]);

                // Log the TrxCode as found
                $rowStatus['trx_codes'][] = [
                    'header'  => $header,
                    'status'  => 'exists',
                    'message' => "TrxCode '{$header}' found and saved."
                ];
            } else {
                // If TrxCode not found, log the error
                $missingTrxMessage = "TrxCode Not Found for Header: '{$header}'";
                Log::error($missingTrxMessage);

                // Add the missing TrxCode status
                $rowStatus['trx_codes'][] = [
                    'header'  => $header,
                    'status'  => 'not_found',
                    'message' => $missingTrxMessage
                ];

                // Mark the row as a warning if the TrxCode is missing
                $rowStatus['status'] = 'warning';
            }
        }


        // Save row status
        $this->failedRows[] = $rowStatus;

        return $ptsr;
    }

    public function getFailedRows()
    {
        return array_map(function ($row) {
            return [
                'row_data' => $row['row_data'],
                'status'   => $row['status'],
                'errors'   => !empty($row['errors']) ? implode(', ', $row['errors']) : 'No errors',
                'trx_codes' => array_map(function ($trx) {
                    return "{$trx['header']}: {$trx['message']}";
                }, $row['trx_codes']),
            ];
        }, $this->failedRows);
    }
}
